(* SmeftFR v2.00 package *)
(* Calculation of Feynman rules for gauge and Higgs couplings in
physical basis *)

TwoFermionFlavorRotate = Function[{x},
(* 2 fermion current flavor rotation and expansion *)
Block[{k,tmp},

tmp = ReplaceFlavorRotations[x // Expand] // Expand;
tmp /. Lam^k_ -> If[k > 1, 0, Lam]

]
(* end of TwoFermionFlavorRotate *)
]



LeptonEWInteractions = Function[{}, 
(* 2-lepton EW couplings *)
Block[{ a, b, k, tmp, LGG, LGH},

(* leptonic vertices only *)
tmp = SMEFT$LGferm /. G[a_,b_] -> 0 /. uq[___] -> 0 /. dq[___] -> 0;
(* express Yukawa's in term of fermion masses *)
tmp = tmp /. yl[a_,b_] -> SMEFT$YL[a,b] // FunctionExpand;
(* pure gauge vertices *)
LGG = tmp /. H -> 0 /. G0->0 /. GP->0 /. GPbar->0;
(* gauge-Higgs part *)
LGH = tmp - LGG;

LeptonGaugeLagrangian      = TwoFermionFlavorRotate[LGG] /.
     Conjugate[fml[a_, b_]] -> mldiag[a] IndexDelta[a,b] /.
               fml[a_, b_]  -> mldiag[a] IndexDelta[a,b];

LeptonHiggsGaugeLagrangian = TwoFermionFlavorRotate[LGH] /.
     Conjugate[fml[a_, b_]] -> mldiag[a] IndexDelta[a,b] /.
               fml[a_, b_]  -> mldiag[a] IndexDelta[a,b];

Print[Style["Calculating leptonic EW vertices...",Bold]];
LeptonGaugeVertices = FeynmanRules[LeptonGaugeLagrangian] // Simplify;
LeptonHiggsGaugeVertices = FeynmanRules[LeptonHiggsGaugeLagrangian] // Simplify;

];
(* end of LeptonEWInteractions*)
];


QuarkEWInteractions = Function[{}, 
(* 2-quark EW couplings *)
Block[{ a, b, k, tmp, LGG, LGH},

(* quark vertices only *)
tmp = SMEFT$LGferm /. G[a_,b_] -> 0 /. l[___] -> 0 /. vl[___] -> 0;
(* express Yukawa's in term of fermion masses *)
tmp = tmp /. yu[a_,b_] -> SMEFT$YU[a,b];
tmp = tmp /. yd[a_,b_] -> SMEFT$YD[a,b] // FunctionExpand;
(* pure gauge vertices *)
LGG = tmp /. H -> 0 /. G0->0 /. GP->0 /. GPbar->0;
(* gauge-Higgs part *)
LGH = tmp - LGG;

QuarkGaugeLagrangian      = TwoFermionFlavorRotate[LGG] /.
     Conjugate[fmd[a_, b_]] -> mddiag[a] IndexDelta[a,b] /.
               fmd[a_, b_]  -> mddiag[a] IndexDelta[a,b] /.
     Conjugate[fmu[a_, b_]] -> mudiag[a] IndexDelta[a,b] /.
               fmu[a_, b_]  -> mudiag[a] IndexDelta[a,b];

QuarkHiggsGaugeLagrangian = TwoFermionFlavorRotate[LGH] /.
     Conjugate[fmd[a_, b_]] -> mddiag[a] IndexDelta[a,b] /.
               fmd[a_, b_]  -> mddiag[a] IndexDelta[a,b] /.
     Conjugate[fmu[a_, b_]] -> mudiag[a] IndexDelta[a,b] /.
               fmu[a_, b_]  -> mudiag[a] IndexDelta[a,b];
      
Print[Style["Calculating quark EW vertices...",Bold]];
QuarkGaugeVertices = FeynmanRules[QuarkGaugeLagrangian] // Simplify;
QuarkHiggsGaugeVertices = FeynmanRules[QuarkHiggsGaugeLagrangian] // Simplify;

];
(* end of QuarkEWInteractions*)
];




QuarkQCDInteractions = Function[{}, 
(* 2-fermion QCD couplings *)
Block[{a,b,k,tmp,LGlF},

tmp =  SMEFT$LGferm /. G[a_,b_]->0;
LGlF = SMEFT$LGferm - tmp // FunctionExpand;

QuarkGluonLagrangian = TwoFermionFlavorRotate[LGlF // Expand];

Print[Style["Calculating quark QCD vertices...",Bold]];
QuarkGluonVertices =  FeynmanRules[QuarkGluonLagrangian] // Simplify;

];
(* end of QuarkQCDInteractions*)
];



(* sort out double defined vertices with psi^C-nu in 4-fermion currents *)
SortDeltaLTwoVertices = Function[{x},
Block[{i,vlist,tmp,a,b},

vlist = x /. ToExpression[ SMEFT$MB <>"vv" ][a_, b_] -> - mvdiag[a]/vev^2/Sqrt[Lam] IndexDelta[a,b];
For[i=1, i < Length[vlist] + 1, i++,
  Clear[tmp];
  If[ (vlist[[i]][[1,1,1]] == lcbar) && (vlist[[i]][[1,2,1]] == vl),
    vlist[[i]][[1,1,1]] = vlbar;
    vlist[[i]][[1,2,1]] = l;
    vlist[[i]][[2]] = vlist[[i]][[2]] /. Index[Generation,Ext[1]] -> tmp;
    vlist[[i]][[2]] = vlist[[i]][[2]] /. Index[Generation,Ext[2]] -> Index[Generation,Ext[1]];
    vlist[[i]][[2]] = vlist[[i]][[2]] /. tmp -> Index[Generation,Ext[2]];
  ];
  If[ (vlist[[i]][[1,1,1]] == vlbar) && (vlist[[i]][[1,2,1]] == lc),
    vlist[[i]][[1,1,1]] = lbar;
    vlist[[i]][[1,2,1]] = vl;
    vlist[[i]][[2]] = vlist[[i]][[2]] /. Index[Generation,Ext[1]] -> tmp;
    vlist[[i]][[2]] = vlist[[i]][[2]] /. Index[Generation,Ext[2]] -> Index[Generation,Ext[1]];
    vlist[[i]][[2]] = vlist[[i]][[2]] /. tmp -> Index[Generation,Ext[2]];
  ];
(* extra combinatorial factor for e+e+G-G- and H.c. vertices *)
  If[ ((vlist[[i]][[1,1,1]] == lbar) && (vlist[[i]][[1,2,1]] == lc)) ||
      ((vlist[[i]][[1,1,1]] == lcbar) && (vlist[[i]][[1,2,1]] == l)),
    vlist[[i]][[2]] = 2 vlist[[i]][[2]];
  ];
];

MergeVertices[vlist]

]
(* end of SortDeltaLTwoVertices *)
]


SwapExternalIndices = Function[{x,i1,i2},
(* swap external indeices excluding spin indices *)
Module[{swap,tmp},
 
tmp  = x /. Ext[i1] -> swap;
tmp = tmp /. Ext[i2] -> Ext[i1];
tmp = tmp /. swap -> Ext[i2];
 
tmp = tmp /. Index[Spin,Ext[i1]] -> Index[Spin,swap];
tmp = tmp /. Index[Spin,Ext[i2]] -> Index[Spin,Ext[i1]];
tmp = tmp /. Index[Spin,swap]    -> Index[Spin,Ext[i2]];

tmp

]
(* end of SwapExternalIndices *)
]




SymmetrizeNeutrino2Current = Function[{x},
(* symmetrize vertices for Majorana neutrinos *)
Block[{a,b,c,swap,i,frule,symlist},

symlist = {};

For[i=1, i < Length[x]+1, i++,
  If[ SubsetQ[ Flatten[ Take[ x[[i]][[1]], All, 1] ], {vlbar,vl} ],
    frule = SwapExternalIndices[x[[i]],1,2];
    Clear[swap];
    frule = frule /. TensDot[SlashedP[a_], ProjM][b_,c_] ->  TensDot[SlashedP[a], swap][b,c];
    frule = frule /. TensDot[SlashedP[a_], ProjP][b_,c_] ->  - TensDot[SlashedP[a], ProjM][b,c];
    frule = frule /. TensDot[SlashedP[a_], swap][b_,c_]  ->  - TensDot[SlashedP[a], ProjP][b,c];
    frule = frule /. TensDot[Ga[a_], ProjM][b_,c_] ->  TensDot[Ga[a], swap][b,c];
    frule = frule /. TensDot[Ga[a_], ProjP][b_,c_] ->  - TensDot[Ga[a], ProjM][b,c];
    frule = frule /. TensDot[Ga[a_], swap][b_,c_]  ->  - TensDot[Ga[a], ProjP][b,c];
    AppendTo[symlist,frule];  
  ];
];

MergeVertices[x,symlist]

]
(* end of SymmetrizeNeutrino2Current *)
]



DeltaLTwoInteractions = Function[{}, 
(*  Delta L=2 violating couplings *) 
Block[{ i4f, oplist },

Print[Style["Calculating Delta L=2 violating vertices...",Bold]];
      
DeltaLTwoLagrangian = If[ MemberQ[ SMEFTOperatorList, "vv" ], Sqrt[Lam] LQvv, 0] // Expand;
DeltaLTwoLagrangian = ReplaceFlavorRotations[DeltaLTwoLagrangian] // Expand;
DeltaLTwoLagrangian = DeltaLTwoLagrangian /. Lam^k_->If[k>1,0,Lam^k];
DeltaLTwoVertices = FeynmanRules[DeltaLTwoLagrangian];
(* remove unnecessary vertices with psi^C fields *)
DeltaLTwoVertices = SortDeltaLTwoVertices[DeltaLTwoVertices];
            
];
(* end of DeltaLTwoInteractions *)
];



HiggsEWInteractions = Function[{}, 
(* EW gauge and Higgs couplings *)
Block[{a, b, c, A, B, C, k, l, m, LGGH, ind1, ind2, ind3, lging,
epstmp, lggh },

LGGH = SMEFT$LGeff + SMEFT$LGeff6 /. G[a_,b_] -> 0 // Expand;
LGGH = LGGH /. Lam^k_->If[k>1,0,Lam];
       
GaugeSelfLagrangian = LGGH /. H->0 /. GP->0 /. GPbar->0 /. G0->0;
GaugeHiggsLagrangian = LGGH - GaugeSelfLagrangian// Expand;

Print[Style["Calculating EW gauge-gauge vertices...",Bold]];
GaugeSelfVertices = FeynmanRules[GaugeSelfLagrangian];
Print[Style["Calculating EW gauge+Higgs vertices...",Bold]];
GaugeHiggsVertices = FeynmanRules[GaugeHiggsLagrangian];

(* some manual index reordering/simplification *)
lggh = {GaugeSelfVertices,GaugeHiggsVertices};

ind1 = Unique[lgind];
ind2 = Unique[lgind];
ind3 = Unique[lgind];

lggh = lggh /. Eps -> epstmp;

lggh = lggh /. epstmp[Index[Lorentz,Ext[k_]], a_, b_, c_] FV[A_, a_] FV[B_, b_] FV[C_, c_] -> 
  epstmp[Index[Lorentz,Ext[k]],ind1,ind2,ind3] FV[A,ind1] FV[B,ind2] FV[C,ind3];

lggh = lggh /. epstmp[Index[Lorentz,Ext[k_]], Index[Lorentz,Ext[l_]], a_, b_] FV[A_, a_] FV[B_, b_] -> 
  epstmp[Index[Lorentz,Ext[k]], Index[Lorentz,Ext[l]],ind1,ind2] FV[A,ind1] FV[B,ind2];

lggh = lggh /. epstmp[Index[Lorentz,Ext[k_]], Index[Lorentz,Ext[l_]], Index[Lorentz,Ext[m_]], a_] FV[A_, a_] -> 
  epstmp[Index[Lorentz,Ext[k]], Index[Lorentz,Ext[l]], Index[Lorentz,Ext[m]],ind1] FV[A,ind1];

lggh = lggh /. ind1-> Index[Lorentz,\[Alpha]1] /. ind2-> Index[Lorentz,\[Beta]1] /. ind3-> Index[Lorentz,\[Gamma]1];

lggh = lggh /. epstmp -> Eps;

GaugeSelfVertices = lggh[[1]];
GaugeHiggsVertices = lggh[[2]];

];
(* end of HiggsEWInteractions *)
];




HiggsQCDInteractions = Function[{}, 
(* QCD gauge and Higgs couplings *)
Block[{a, b, c, A, B, C, k, l, m, tmp, LGall, LGluon, lgind, ind1,
ind2, ind3, epstmp},

LGall = SMEFT$LGeff + SMEFT$LGeff6 // FunctionExpand;
tmp = LGall /. G[a_,b_]->0;
LGluon = LGall - tmp // Expand;
LGluon = LGluon /. Lam^k_ -> If[k>1,0,Lam^k];

GluonSelfLagrangian = LGluon /. H->0 /. GP->0 /. GPbar->0 /. G0->0;
GluonHiggsLagrangian = LGluon - GluonSelfLagrangian // Expand;
      
Print[Style["Calculating QCD gauge+Higgs vertices...",Bold]];
GluonSelfVertices  = FeynmanRules[GluonSelfLagrangian];
GluonHiggsVertices = FeynmanRules[GluonHiggsLagrangian];

(* manual index simplifications *)      
tmp = {GluonSelfVertices,GluonHiggsVertices};

ind1 = Unique[lgind];
ind2 = Unique[lgind];
ind3 = Unique[lgind];

tmp = tmp /. Eps -> epstmp;

tmp = tmp /. epstmp[Index[Lorentz,Ext[k_]], a_, b_, c_] FV[A_, a_] FV[B_, b_] FV[C_, c_] -> 
  epstmp[Index[Lorentz,Ext[k]],ind1,ind2,ind3] FV[A,ind1] FV[B,ind2] FV[C,ind3];

tmp = tmp /. epstmp[Index[Lorentz,Ext[k_]], Index[Lorentz,Ext[l_]], a_, b_] FV[A_, a_] FV[B_, b_] -> 
  epstmp[Index[Lorentz,Ext[k]], Index[Lorentz,Ext[l]],ind1,ind2] FV[A,ind1] FV[B,ind2];

tmp = tmp /. epstmp[Index[Lorentz,Ext[k_]], Index[Lorentz,Ext[l_]], Index[Lorentz,Ext[m_]], a_] FV[A_, a_] -> 
  epstmp[Index[Lorentz,Ext[k]], Index[Lorentz,Ext[l]], Index[Lorentz,Ext[m]],ind1] FV[A,ind1];

tmp = tmp /. ind1-> Index[Lorentz,\[Alpha]1] /. ind2-> Index[Lorentz,\[Beta]1] /. ind3-> Index[Lorentz,\[Gamma]1];

tmp = tmp /. epstmp -> Eps;

GluonSelfVertices = tmp[[1]];
GluonHiggsVertices = tmp[[2]];

];
(* end of HiggsQCDInteractions *)
];


