(* SmeftFR v2.00 package *)
(* routines for help interfaces *)

SMEFTInitializeMB[ OptionsPattern[{ RealParameters -> True,
                                    Include4Fermion -> False(*, 
                                    IncludeBL4Fermion -> False*) }] ] := 
(* Load MB parameter files and Lagrangian *)
Block[{ism, jsm, vv, zm, gw, tmp, L2Fermion, LEW, LQCD, L4Fermion,
GhostLagrangian, LSMEFT},

(* Load SM fields and parameter definitions in mass basis*)
Get[ FileNameJoin[{SMEFT$Path, "definitions", "smeft_fields_MB.fr"}] ];
If [ OptionValue[RealParameters] === False,
  Get[ FileNameJoin[{SMEFT$Path, "output", "smeft_par_MB.fr"}] ];
, 
  Get[ FileNameJoin[{SMEFT$Path, "output", "smeft_par_MB_real.fr"}] ];
];
LoadModel[];

(* load MB Lagrangian *)
Get[ FileNameJoin[{SMEFT$Path, "output", "smeft_feynman_rules.m"}] ];

vv = NumericalValue[vev];
zm = NumericalValue[MZ];
gw = NumericalValue[GW];

(* truncate imaginary part of CKM and PMNS if RealParameters -> True *)
If [ OptionValue[RealParameters] === True,
  For[ism=1, ism<4, ism++,
    For[jsm=1, jsm<4, jsm++,
      UpdateParameters[ K[ism,jsm] -> CKM$real[[ism,jsm]] ];
      UpdateParameters[ U[ism,jsm] -> PMNS$real[[ism,jsm]] ];
    ]; 
  ]; 
];

(* update Higgs quartic coupling *)
tmp = NumericalValue[lam];
tmp = tmp + (If[ MemberQ[ SMEFTOperatorList, "phi" ], 
        3 NumericalValue[ ToExpression[ SMEFT$MB <> "phi" ] ], 0] +
             If[ MemberQ[ SMEFTOperatorList, "phiBox" ], 
       -2 NumericalValue[ ToExpression[ SMEFT$MB <> "phiBox" ] ], 0] tmp +  
             If[ MemberQ[ SMEFTOperatorList, "phiD" ], 
          NumericalValue[ ToExpression[ SMEFT$MB <> "phiD" ] ]/2, 0] tmp) vv^2;
UpdateParameters[ lam -> tmp ];                

(* update g' = G1 coupling *)
tmp = - 2 zm/vv + 
        vv zm/2 If[ MemberQ[ SMEFTOperatorList, "phiD" ], 
          NumericalValue[ ToExpression[ SMEFT$MB <> "phiD" ] ], 0 ] + 
        gw vv^2/2/zm Sqrt[4 zm^2 - gw^2 vv^2] If[ MemberQ[ SMEFTOperatorList, "phiWB" ], 
          NumericalValue[ ToExpression[ SMEFT$MB <> "phiWB" ] ], 0 ];
UpdateParameters[ G1 -> Sqrt[ tmp^2 - gw^2 ] ];
       
(* update neutrino mass parameters *)
UpdateParameters[ mve->NumericalValue[v1mass], 
                  mvm->NumericalValue[v2mass],
                  mvt->NumericalValue[v3mass] ];

(* update ghost and Goldstone masses in Rxi gauge, unused in unitary gauge *)
UpdateParameters[ MgZ -> Sqrt[NumericalValue[xiZ]] zm,
                  MG0 -> Sqrt[NumericalValue[xiZ]] zm ];
UpdateParameters[ MgW -> Sqrt[NumericalValue[xiW]] NumericalValue[MW],
                  MGP -> Sqrt[NumericalValue[xiW]] NumericalValue[MW] ];

L2Fermion = LeptonGaugeLagrangian + LeptonHiggsGaugeLagrangian +
    QuarkGaugeLagrangian + QuarkHiggsGaugeLagrangian;
LEW  = GaugeSelfLagrangian + GaugeHiggsLagrangian;
LQCD = QuarkGluonLagrangian + GluonSelfLagrangian + GluonHiggsLagrangian;

LSMEFT = L2Fermion + LEW + LQCD + If[ SMEFTRxiGaugeStatus, GhostLagrangian, 0];

(* 4-fermion vertices may take very long time to calculate and result
may not be correct due to relative vertex sign problems!  *)
If [ OptionValue[Include4Fermion] === True,  
  L4Fermion = FourLeptonLagrangian + 
              TwoQuarkTwoLeptonLagrangian + 
              FourQuarkLagrangian (* +
              If [ OptionValue[IncludeBL4Fermion] === True, 
                BLViolatingLagrangian, 0] *);
  LSMEFT = LSMEFT + L4Fermion;
];

Clear[ism];
(* set proper interaction order for Yukawa dependent terms *)
LSMEFT = LSMEFT /. mvdiag[ism_] -> YO mvdiag[ism] /. 
                   mldiag[ism_] -> YO mldiag[ism] /. 
                   mddiag[ism_] -> YO mddiag[ism] /.
                   mudiag[ism_] -> YO mudiag[ism];
(* scale included in WC's, so set to 1 here *)
SMEFTMBLagrangian = LSMEFT /. Lam -> 1;

]
(* end of SMEFTInitializeMB *)




SMEFTToWCXF[ SourceFile_, TargetFile_ ] :=
(* WCxf JSON export function - reads from SMEFT model file and generates Wilson coefficients file in JSON format  *)
Block[{il,it,WC,WN,ig1,ig2,ig3,ig4,tmp,date,values,output,oplist,first,temp,mspar},

If [ ! FileExistsQ[ SourceFile ],
  Print["File " <> SourceFile <> " does not exist, check its name and location"];
  Abort[];
];

Get[ SourceFile ];

(* find parameters with "Value" assigned *)
mspar = {};
temp = Value /. Values[ Rule @@@ M$Parameters ];
For[il=1, il < Length[temp] + 1, il++,
  If[ temp[[il]] =!= Value, AppendTo[ mspar, (Rule @@@ M$Parameters)[[il]] ] ];
];

(* find dim-5 and 6 WC names only, skip other parameters *)
temp = Complement[ ToString /@ Keys[mspar],  {"GW","G1","GS", "K", "U"} ];
oplist = Intersection[ SMEFTAllOperators, StringDrop[ temp, 1 ] ];

(* try to identify first letter in WCs definition *)
first = "";
alph = Join[ToUpperCase /@ Alphabet[], Alphabet[]];
For[il=1, il < Length[alph] + 1, il++,
  If[ Length[ Intersection[temp, StringInsert[ oplist, alph[[il]], 1 ] ] ] != 0, first = alph[[il]] ];
];
If [ first == "",
  Print[Style["Warning: parameter file does not contain valid Wilson coefficients",Red]]; 
  oplist = {}; 
];

ActiveS = Intersection[ToString /@ ScalarWC, oplist];
ScalarVal = If[ Length[ActiveS] > 0, Value /. (ToExpression[StringInsert[ActiveS, first, 1]] /. mspar), 0 ];

Active2T = Intersection[Tensor2WC[[All,1]], oplist];
Tensor2Val = If[ Length[Active2T] > 0, Association[ Flatten[ Value /. (ToExpression[StringInsert[Active2T, first, 1]] /. mspar) ] ], 0 ];

Active4T = Intersection[Tensor4WC[[All,1]], oplist];
Tensor4Val = If[ Length[Active4T] > 0, Association[ Flatten[ Value /. (ToExpression[StringInsert[Active4T, first, 1]] /. mspar) ] ], {} ];

Print["Wilson coefficients in WCXF format stored as ", Style[ TargetFile, Bold ] ];

output = {};
      
date = Date[];
AppendTo[ output, "eft" -> "SMEFT" ];
AppendTo[ output, "basis" -> "Warsaw mass" ];
AppendTo[ output, "metadata" -> { "generator" -> " SmeftFR v" <> SMEFT$Version <> ", SMEFTToWCXF routine",
			          "description" -> "Generated from the FeynRules model file " <>
	                           SourceFile <> " on " <> ToString[date[[3]]] <> "." <>
	                           ToString[date[[2]]] <> "." <> ToString[date[[1]]] <> " " <>
	                           ToString[date[[4]]] <> ":" <> ToString[date[[5]]] } ];

temp = If [ MemberQ[ Keys[mspar], Lam ], Sqrt[Value /. (Lam /. mspar)], Sqrt[Value] ];
AppendTo[ output, "scale" -> If[ NumberQ[temp], temp, 0 ] ];

values = {};
      
For[ il=1, il < Length[ActiveS] + 1, il++,
  If[ NumberQ[ ScalarVal[[il]]], AppendTo[ values, (ActiveS[[il]] /. ScalarJSON) -> ScalarVal[[il]] ] ];
];

Active2T = Complement[Active2T, {"vv"}];
For[ il=1, il < Length[Active2T] + 1, il++,
  tmp = Tensor2Ind[ Active2T[[il]] ];
  For[ it=1, it < Length[tmp] + 1, it++,
    ig1 = tmp[[it,1]];
    ig2 = tmp[[it,2]];
    WN = (Active2T[[il]] /. Tensor2JSON) <> "_" <> ToString[ig1] <> ToString[ig2];
    WC = Tensor2Val[ ToExpression[ first <> Active2T[[il]] ] @@ {ig1,ig2} ];
    If[ NumberQ[WC],
      If[ tmp[[it,3]] && NumberQ[WC],
        AppendTo[ values, WN ->  WC];
      ,	   
        AppendTo[ values, WN -> {"Re" -> Re[WC], "Im" -> Im[WC]} ];
      ];
    ];
  ];
];

For[ il=1, il < Length[Active4T] + 1, il++,
  tmp = Tensor4Ind[ Active4T[[il]] ];
  For[ it=1, it < Length[tmp] + 1, it++,
    ig1 = tmp[[it,1]];
    ig2 = tmp[[it,2]];
    ig3 = tmp[[it,3]];
    ig4 = tmp[[it,4]];
    WN = (Active4T[[il]] /. Tensor4JSON) <> "_" <> ToString[ig1] <> ToString[ig2] <> ToString[ig3] <> ToString[ig4];
    WC = Tensor4Val[ ToExpression[ first <> Active4T[[il]] ] @@ {ig1,ig2,ig3,ig4} ];
    If[ NumberQ[WC],
      If[ tmp[[it,5]],
        AppendTo[ values, WN ->  WC];
      ,	   
        AppendTo[ values, WN -> {"Re" -> Re[WC], "Im" -> Im[WC]} ];
      ];
    ];
  ];
];

(* special treatment for vv operator *)
If[ MemberQ[oplist, "vv"],
  tmp = Tensor2Ind[ "vv" ];
  For[ it=1, it < Length[tmp] + 1, it++,
    ig1 = tmp[[it,1]];
    ig2 = tmp[[it,2]];
    WN = ("vv" /. Tensor2JSON) <> "_" <> ToString[ig1] <> ToString[ig2];
    WC = Tensor2Val[ ToExpression[ first <> "vv"] @@ {ig1,ig2} ];
    AppendTo[ values, WN -> {"Re" -> Re[WC], "Im" -> Im[WC]} ];
  ];
];

AppendTo[output, "values" -> values ];
Export[TargetFile, output, "json"];
      
]
(* end of SMEFTToWCXF *)




WCXFToSMEFT[ SourceFile_, TargetFile_, 
             OptionsPattern[ {Operators -> SMEFTAllOperators, 
                              OverwriteTarget->False, 
                              RealParameters->True, 
                              Silent->False, 
                              FirstLetter->"C"} ] ] :=
(* WCxf JSON import function - reads from JSON format and generates SMEFT model file *)
Module[{JSONData,JSONWilson,il,WC,ig1,ig2,ig3,ig4,tmp,date,cfile,TargetExistsQ,WC2,WC4,ActiveS,Active2T,Active4T,oplist},
      
WCXF$MB = OptionValue[FirstLetter];
If [ Head[ WCXF$MB ] === String && StringLength[ WCXF$MB ] > 0, 
  WCXF$MB = StringTake[WCXF$MB,1];
,
  Print["Non-string value for FirstLetter, check selected WCXFToSMEFT options"];
  Abort[];
];

If[ Head[SourceFile] =!= String ||  Head[TargetFile] =!= String,
  Print["Source and target file names must be strings!"];
  Abort[];
];

If [ ! FileExistsQ[ SourceFile ],
  Print["File " <> SourceFile <> " does not exist, check its name and location"];
  Abort[];
];

If [ FileFormat[ SourceFile ] != "JSON",
  Print["File " <> SourceFile <> " not in JSON format or does not have extension .json or .JSON, please correct!"];
  Abort[];
];

oplist = OptionValue[Operators];
If[ Head[oplist] =!= List, Print["Option 'Operators' must be a list, aborting!"];Abort[];];

If [ OptionValue[Silent] =!= True, Print["Wilson coefficient values initialized from WCxf file ", Style[ SourceFile, Bold ] ] ];
JSONData = Import[ SourceFile ];
JSONWilson = SMEFTToComplex /@ Association["values" /. JSONData];

ActiveS = ToString /@ Intersection[ScalarWC, oplist];
ScalarVal = {};
For[ il=1, il < Length[ActiveS] + 1, il++,
  WC = JSONWilson[ ActiveS[[il]] /. ScalarJSON ];
  AppendTo[ ScalarVal, ActiveS[[il]] -> If[NumberQ[WC], If[ OptionValue[RealParameters], Re[WC], WC], 0] ];
];

Active2T = ToString /@ Intersection[ Tensor2WC[[All,1]], oplist ];
Tensor2Val = {};
For[ il = 1, il < Length[Active2T] + 1, il++,
  WC2 = Table[0,{ig1,1,3},{ig2,1,3}];    
  For[ ig1 = 1, ig1 < 4, ig1++,	  
    For[ ig2 = 1, ig2 < 4, ig2++,
      WC = JSONWilson[ (Active2T[[il]] /. Tensor2JSON) <> "_" <> ToString[ig1] <> ToString[ig2] ];
      WC2[[ig1,ig2]] = If[ NumberQ[WC], If[ OptionValue[RealParameters], Re[WC], WC], 0];	 
    ];
  ];
  WC2 = Tensor2WCSymmetrize[ WC2, ToExpression[ Active2T[[il]] ] /. Tensor2Class ];
  tmp = {}; 
  For[ ig1 = 1, ig1 < 4, ig1++,	  
    For[ ig2 = 1, ig2 < 4, ig2++,
      AppendTo[ tmp, Active2T[[il]][ig1,ig2] -> WC2[[ig1,ig2]] ];
    ];
  ];
  AppendTo[Tensor2Val, Active2T[[il]] -> tmp];
];


Active4T = ToString /@ Intersection[Tensor4WC[[All,1]], oplist];
Tensor4Val = {};
For[ il = 1, il < Length[Active4T] + 1, il++,
  WC4 = Table[0,{ig1,1,3},{ig2,1,3},{ig3,1,3},{ig4,1,3}];    
  For[ ig1 = 1, ig1 < 4, ig1++,
    For[ ig2 = 1, ig2 < 4, ig2++,
      For[ ig3 = 1, ig3 < 4, ig3++,
        For[ ig4 = 1, ig4 < 4, ig4++,
	  WC = JSONWilson[ (Active4T[[il]] /. Tensor4JSON)  <> "_" <>
	    ToString[ig1] <> ToString[ig2] <> ToString[ig3] <> ToString[ig4] ];
	  WC4[[ig1,ig2,ig3,ig4]] = If[ NumberQ[WC], If[ OptionValue[RealParameters], Re[WC], WC], 0];
        ];
      ];
    ];
  ];
  WC4 = Tensor4WCSymmetrize[ WC4, ToExpression[ Active4T[[il]] ] /. Tensor4Class ];
  tmp = {};
  For[ ig1 = 1, ig1 < 4, ig1++,
    For[ ig2 = 1, ig2 < 4, ig2++,
      For[ ig3 = 1, ig3 < 4, ig3++,
        For[ ig4 = 1, ig4 < 4, ig4++,
          AppendTo[ tmp, Active4T[[il]][ig1,ig2,ig3,ig4] -> WC4[[ig1,ig2,ig3,ig4]] ];
        ];
      ];
    ];
  ];
  AppendTo[Tensor4Val, Active4T[[il]] -> tmp];
];

If [ FileExistsQ[ TargetFile ] && (! OptionValue[OverwriteTarget]),
    Print["File " <> TargetFile <> " exists!"];
    TargetExistsQ = AskFunction[ Ask["Overwrite (y/n)?"-> "String"] ][];
];
If[ TargetExistsQ != "y", Print[ "Change name of parameter file generated from WCXF format and rerun" ]; Abort[]; ];
If [ OptionValue[Silent] =!= True,
  Print["Parameter file in mass basis generated as ",  Style[TargetFile, Bold ] ]; 
  If[ OptionValue[RealParameters], Print["(it contains only real parts of WC's, as accepted by MADGRAPH5; use option RealParameters->False to force complex parameters)" ] ];
];
  
cfile = OpenWrite[TargetFile];

WriteLine[cfile, "(* Generated by WCXFToSMEFT routine from the file " <> SourceFile <> " *)"];
date = Date[];
WriteLine[cfile, "(* " <> ToString[date[[3]]] <> "." <>
    ToString[date[[2]]] <> "." <> ToString[date[[1]]] <> " " <>
    ToString[date[[4]]] <> ":" <> ToString[date[[5]]] <> " *)\n\n"]
WriteLine[cfile, "(* Active operators included in Feynman Rules: " <> ToString[oplist] <> " *)\n"];
      
WriteString[cfile, ReadString[FileNameJoin[{SMEFT$Path, "definitions", "smeft_par_head_MB.fr"}]] ];
WriteLine[cfile, "\n"];


WriteLine[cfile, "(* neutrino masses *)\n"];
For[ig1 = 1, ig1 < 4, ig1++,
  If[ MemberQ[ oplist, "vv"], 
    WC = JSONWilson[ ("vv" /. Tensor2JSON) <> "_" <> ToString[ig1] <> ToString[ig1] ];
    tmp = If[ NumberQ[WC], Abs[WC]/Sqrt[2]/SMEFT$Gf, 0];
  ,	
    tmp = 0;
  ];
  WriteLine[cfile, "  v" <> ToString[ig1] <> "mass == {"];
  WriteLine[cfile, "    ParameterType    -> External,"];
  WriteLine[cfile, "    Value            -> " <> NumberToString[tmp] <> ","];
  WriteLine[cfile, "    Description      -> \"v" <> ToString[ig1] <> " mass\""];
  WriteLine[cfile, "  },"];
];
WriteLine[cfile, ""];

WriteLine[cfile, "(* redefined (mass basis) dim-6 couplings *)\n"];
WriteLine[cfile, "(* flavor independent *)\n"];
For[ il=1, il < Length[ActiveS] + 1,  il++, WriteExternalScalarEntry[ cfile, ScalarVal[[il]] ]; ];

WriteLine[cfile, "(* flavor dependent *)\n"];
WriteLine[cfile, "(* 2 fermion operators *)\n"];
For[ il=1, il < Length[Active2T] + 1, il++, WriteExternalTensor2Entry[ cfile, Tensor2Val[[il]] ]; ];
WriteLine[cfile, "(* 4 fermion operators *)\n"];
For[ il=1, il < Length[Active4T] + 1, il++, WriteExternalTensor4Entry[ cfile, Tensor4Val[[il]] ]; ];

WriteLine[cfile, "(* Effective NP scale squared*)\n"];
WriteLine[cfile, "  Lam == {"];
WriteLine[cfile, "    ParameterType -> External,"];
WriteLine[cfile, "    Value   -> " <> NumberToString[ ("scale" /. JSONData)^2 ] <> ","];
WriteLine[cfile, "    TeX           -> 1/\[CapitalLambda]^2,"];
WriteLine[cfile, "    Description   -> \"Effective NP scale squared\""];
WriteLine[cfile, "  }\n\n"];

WriteLine[cfile, "};\n"];

Close[TargetFile];

]
(* end of WCXFToSMEFT *)




WriteExternalScalarEntry = Function[{cfile,wc},
(* par file entry for scalar mass basis external WC *)
(* wc argument is association wcname->wcvalue(s) *)
Module[{tex,name,pos},

name = ToString[ Keys[wc] ];
pos = StringPosition[name, "phi",1];
tex = name;
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];      

WriteLine[cfile, " " <> WCXF$MB <> name <> " == {"];
WriteLine[cfile, "    ParameterType    -> External,"];
WriteLine[cfile, "    Indices          -> {},"];
WriteLine[cfile, "    InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "    Value            -> " <> NumberToString[Keys[wc] /. wc] <> "," ];
WriteLine[cfile, "    ComplexParameter -> False,"];
WriteLine[cfile, "    TeX              -> Superscript[" <> WCXF$MB <> ", " <> tex <> "],"]; 
WriteLine[cfile, "    Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "    },\n\n"];

]
(* end of WriteExternalScalarEntry *)			 
]



WriteExternalTensor2Entry = Function[{cfile,wc},
(* par file entry for 2-fermion mass basis external WC *)
(* wc argument is association wcname->wcvalue(s) *)
Module[{WC,tex,name,pos,i,wcname,wcval,tmp},

name = ToString[ Keys[wc] ];
WC = wc /. (Keys[wc] -> ToExpression[WCXF$MB <> name]);
WC = Keys[WC] /. WC;
tex = name;
pos = StringPosition[tex, "phi",1];
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];      
pos = StringPosition[tex, "f",1];
If [Length[StringPosition[tex, "f"]]   != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

wcname = ToString /@ Keys[WC];
wcval  = NumberToString /@ Values[WC];

tmp = "    Value            -> {" <> wcname[[1]] <> " -> " <> wcval[[1]] <> ",\n";
For[i=2, i < Length[wcname], i++,
  tmp = tmp <> "                         " <> wcname[[i]] <> " -> " <> 
                                              wcval[[i]]  <> ",\n";
];
tmp = tmp <> "                         " <> wcname[[Length[wcname]]] <> 
             " -> " <> wcval[[Length[wcname]]] <> "},";

WriteLine[cfile, " " <> WCXF$MB <> name <> " == {"];
WriteLine[cfile, "    ParameterType    -> External,"];
WriteLine[cfile, "    Indices          -> {Index[Generation], Index[Generation]},"];
WriteLine[cfile, "    InteractionOrder -> {NP,1},"];  
WriteLine[cfile, tmp];
WriteLine[cfile, "    Hermitian        -> " <> If[ (Keys[wc] /. Tensor2Class) == 2, "True", "False" ] <> ","];
WriteLine[cfile, "    TeX              -> Superscript[" <> WCXF$MB <> ", " <> tex <> "],"]; 
WriteLine[cfile, "    Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "    },\n\n"];

]
(* end of WriteExternalTensor2Entry *)			    
]



WriteExternalTensor4Entry = Function[{cfile,wc},
(* par file entry for 4-fermion mass basis external WC *)
(* wc argument is association wcname->wcvalue(s) *)
Module[{name,i,wcname,wcval,tmp},

name = ToString[ Keys[wc] ];
WC = wc /. (Keys[wc] -> ToExpression[WCXF$MB <> name]);
WC = Keys[WC] /. WC;

wcname = ToString /@ Keys[WC];
wcval  = NumberToString /@ Values[WC];

tmp = "    Value            -> {" <> wcname[[1]] <> " -> " <> wcval[[1]] <> ",\n";
For[i=2, i < Length[wcname], i++,
  tmp = tmp <> "                         " <> wcname[[i]] <> " -> " <> 
                                              wcval[[i]]  <> ",\n";
];
tmp = tmp <> "                         " <> wcname[[Length[wcname]]] <> 
             " -> " <> wcval[[Length[wcname]]] <> "},";

WriteLine[cfile, " " <> WCXF$MB <> name <> " == {"];
WriteLine[cfile, "    ParameterType    -> External,"];
WriteLine[cfile, "    Indices          -> {Index[Generation], Index[Generation], Index[Generation], Index[Generation]},"];
WriteLine[cfile, "    InteractionOrder -> {NP,1},"];  
WriteLine[cfile, tmp];
WriteLine[cfile, "    TeX              -> Superscript[" <> WCXF$MB <> ", " <> name <> "],"]; 
WriteLine[cfile, "    Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "    },\n\n"];

]
(* end of WriteExternalTensor4Entry *)		    
]


