(* SmeftFR v2.0 package *)
(* I/O routines: code for generating FeynRules parameter files with
internal and external parameters, WCxf import and export routines *)


SMEFTPackageInfo = Function[{},
(* print package information to screen *)

Print["*****************************************"];
Print[" - SmeftFR package - "];
Print["Version:    ", SMEFT$Version, " ("SMEFT$Date, ")"];
Print["Authors:    A. Dedes, M. Paraskevas, J. Rosiek, K. Suxho, L. Trifyllis"];
Print["URL:        www.fuw.edu.pl/smeft"];
Print["Maintainer: janusz.rosiek@fuw.edu.pl"];
Print[" "];
Print["Please cite:"];
Print["    - JHEP 1706 (2017) 143, arXiv:1704.03888"]; 
Print["    - arXiv:1904.03204"];
Print["*****************************************"];
Print[" "];

(* Comput.Phys.Commun. XXX (2019) YYY-ZZZ *)

(* end of SMEFTPackageInfo *)
]



SMEFTSetDirectory == Function[{Major,Minor},
(* construct SMEFT package directory and check if it exists *)

SMEFT$Path = FileNameJoin[{$FeynRulesPath, "Models", "SMEFT_" <> 
                ToString[Major] <> "_" <> ToString[Minor]}];
If[ ! DirectoryQ[SMEFT$Path], 
  Print["Directory " <> SMEFT$Path <> "does not exist, please check package setup"];
  Abort[];
]

Print[SMEFT$Path]

]



SMEFTInitializeModel[ OptionsPattern[{ Operators -> SMEFTAllOperators,
                         Gauge -> "Unitary",
                         WCXFInitFile -> "", 
                         MajoranaNeutrino -> False, 
                         WBFirstLetter -> "c", 
                         MBFirstLetter -> "C",
                         Correct4Fermion -> True }] ] := 
Block[{WCXFImportFile},
(* routines generating and loading FeynRules parameter files and SMEFT Lagrangian *)

(* naming convention - editable first letter of WC names in Warsaw and
mass basis. Defaults are c and C, respectively: *)
SMEFT$WB = OptionValue[WBFirstLetter];
If [ Head[ SMEFT$WB ] === String && StringLength[ SMEFT$WB ] > 0, 
  SMEFT$WB = StringTake[SMEFT$WB,1];
,
  Print["Non-string value for WBFirstLetter, check selected SMEFTInitializeModel options"];
  Abort[];
];
SMEFT$MB = OptionValue[MBFirstLetter];
If [ Head[ SMEFT$MB ] === String && StringLength[ SMEFT$MB ] > 0, 
  SMEFT$MB = StringTake[SMEFT$MB,1];
,
  Print["Non-string value for MBFirstLetter, check selected SMEFTInitializeModel options"];
  Abort[];
];

(* active operator list in string and expression formats *)
SMEFTOperatorList = Intersection[ToString /@ OptionValue[Operators], SMEFTAllOperators];
  
SMEFTRxiGaugeStatus = TrueQ[ ToString[OptionValue[Gauge]] === "Rxi" || ToString[OptionValue[Gauge]] === "rxi" ];

SMEFTMajoranaNeutrino = MemberQ[SMEFTOperatorList, "vv"] || TrueQ[ OptionValue[MajoranaNeutrino] === True ];

SMEFTCorrect4FermionSign = OptionValue[Correct4Fermion];

WCXFImportFile = ToString[ OptionValue[WCXFInitFile] ];
If [ WCXFImportFile != "",
  If[ ! FileExistsQ[WCXFImportFile], 
    Print["File " <> WCXFImportFile <> " does not exist, Wilson coefficient values will be initialized to 0"];
    WCXFImportFile = FileNameJoin[{SMEFT$Path, "definitions", "wcxf_default.json"}];
  ];
,
  WCXFImportFile = FileNameJoin[{SMEFT$Path, "definitions", "wcxf_default.json"}];
];

(* setup printout *)
Print["Operators included in Feynman rules: ", Style[ SMEFTOperatorList, Bold ] ];
Print["Feynman rules will be calculated for ", 
  Style[ If[SMEFTRxiGaugeStatus,"R_xi","unitary"] <> " gauge", Bold ] ];
Print["Neutrinos are treated as ", 
      Style[If[SMEFTMajoranaNeutrino, "Majorana", "massless Weyl"], Bold ], " spinors\n" ];

(* generate parameter files in the Warsaw and mass basis *)      
SMEFTParFile[SMEFTOperatorList, FileNameJoin[{SMEFT$Path, "output", "smeft_par_WB.fr"}] ];

WCXFToSMEFT[WCXFImportFile, FileNameJoin[{SMEFT$Path, "output",
      "smeft_par_MB.fr"}], Operators->SMEFTOperatorList,
      OverwriteTarget->True, RealParameters->False, FirstLetter->
      SMEFT$MB ];

WCXFToSMEFT[WCXFImportFile, FileNameJoin[{SMEFT$Path, "output",
      "smeft_par_MB_real.fr"}], Operators->SMEFTOperatorList,
      OverwriteTarget->True, RealParameters->True, Silent->True,
      FirstLetter-> SMEFT$MB ];

Print[];

(* end of SMEFTInitializeModel *)
]


SMEFTParFile = Function[{SMEFTOperatorList, TargetFile},
(* generates FeynRules par file including only "active" operators and internal parameters *)
Block[{il,WC,tmp,date,cfile},
   
Print["Parameter file in Warsaw basis generated as ", Style[ TargetFile, Bold ] ]; 

cfile = OpenWrite[TargetFile];

WriteLine[cfile, "(* Generated by SMEFTParFile routine *)"];
date = Date[];
WriteLine[cfile, "(* " <> ToString[date[[3]]] <> "." <>
    ToString[date[[2]]] <> "." <> ToString[date[[1]]] <> " " <>
    ToString[date[[4]]] <> ":" <> ToString[date[[5]]] <> " *)\n"]

WriteLine[cfile, "(* Active operators included in Feynman Rules: " <> ToString[SMEFTOperatorList] <> " *)\n"];
(*
WriteLine[cfile, "Print[\"Loading SMEFT parameter definitions from \", Style[\"" <> TargetFile <> "\", Bold ] ];\n\n"];
WriteLine[cfile, "Print[];\n\n"];
*)
      
WriteString[cfile, ReadString[FileNameJoin[{SMEFT$Path, "definitions", "smeft_par_head_WB.fr"}]] ];
WriteLine[cfile, "\n"];

WriteLine[cfile, "(* dim 6 couplings in Warsaw basis *)\n"];
WriteLine[cfile, "(* flavor independent *)\n"];
For[ il=1, il < Length[ScalarWC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, ScalarWC[[il]] ], 
    WriteInternalScalarEntry[ cfile, ScalarWC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* flavor dependent *)\n"];
WriteLine[cfile, "(* 2 fermion operators *)\n"];
For[ il=1, il < Length[Tensor2WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor2WC[[il,1]] ] && (! Tensor2WC[[il,5]]), 
    WriteInternalTensor2Entry[ cfile, Tensor2WC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* 4 fermion operators *)\n"];
For[ il=1, il < Length[Tensor4WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor4WC[[il,1]] ] && (! Tensor4WC[[il,6]]), 
    WriteInternalTensor4Entry[ cfile, Tensor4WC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* L or B violating operators *)\n"];
For[ il=1, il < Length[Tensor2WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor2WC[[il,1]] ] && Tensor2WC[[il,5]], 
    WriteInternalTensor2Entry[ cfile, Tensor2WC[[il]] ]; 
  ];
];
For[ il=1, il < Length[Tensor4WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor4WC[[il,1]] ] && Tensor4WC[[il,6]], 
    WriteInternalTensor4Entry[ cfile, Tensor4WC[[il]] ]; 
  ];
];
    
WriteLine[cfile, "(* redefined (mass basis) dim-6 couplings *)\n"];
WriteLine[cfile, "(* flavor independent *)\n"];
For[ il=1, il < Length[ScalarWC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, ScalarWC[[il]] ], 
    WriteInternalMassScalarEntry[ cfile, ScalarWC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* flavor dependent *)\n"];
WriteLine[cfile, "(* 2 fermion operators *)\n"];
For[ il=1, il < Length[Tensor2WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor2WC[[il,1]] ] && (! Tensor2WC[[il,5]]), 
    WriteInternalMassTensor2Entry[ cfile, Tensor2WC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* 4 fermion operators *)\n"];
For[ il=1, il < Length[Tensor4WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor4WC[[il,1]] ] && (! Tensor4WC[[il,6]]), 
    WriteInternalMassTensor4Entry[ cfile, Tensor4WC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* L or B violating operators *)\n"];
For[ il=1, il < Length[Tensor2WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor2WC[[il,1]] ] && (Tensor2WC[[il,5]]), 
    WriteInternalMassTensor2Entry[ cfile, Tensor2WC[[il]] ]; 
  ];
];
For[ il=1, il < Length[Tensor4WC] + 1, il++, 
  If[ MemberQ[SMEFTOperatorList, Tensor4WC[[il,1]] ] && (Tensor4WC[[il,6]]), 
    WriteInternalMassTensor4Entry[ cfile, Tensor4WC[[il]] ]; 
  ];
];

WriteLine[cfile, "(* NP scale *)\n"];
  
WriteLine[cfile, "Lam == {"];
WriteLine[cfile, "  ParameterType -> Internal,"];
WriteLine[cfile, "  TeX           -> 1/\[CapitalLambda]^2,"];
WriteLine[cfile, "  Description   -> \"Effective NP scale squared\""];
WriteLine[cfile, "}\n"];

WriteLine[cfile, "};\n"];

Close[TargetFile];

]
(* end of SMEFTParFile *)
]





(* export routine, store Lagrangian and Feynman in mass basis to file *)
SMEFTOutput[ TargetFile_, OptionsPattern[ OverwriteTarget -> True ] ] :=
Block[{cfile,vlist,laglist,clist,ylist,slist,date,il,ZFlocal,tmp,i,j},
      
vlist = {"LeptonGaugeVertices", "LeptonHiggsGaugeVertices",
"QuarkGaugeVertices", "QuarkHiggsGaugeVertices", "QuarkGluonVertices",
"GaugeSelfVertices", "GaugeHiggsVertices", "GluonSelfVertices",
"GluonHiggsVertices", "FourLeptonVertices",
"TwoQuarkTwoLeptonVertices", "FourQuarkVertices", "BLViolatingVertices"};

laglist = {"LeptonGaugeLagrangian", "LeptonHiggsGaugeLagrangian",
"QuarkGaugeLagrangian", "QuarkHiggsGaugeLagrangian",
"QuarkGluonLagrangian", "GaugeSelfLagrangian", "GaugeHiggsLagrangian",
"GluonSelfLagrangian", "GluonHiggsLagrangian", "FourLeptonLagrangian",
"TwoQuarkTwoLeptonLagrangian", "FourQuarkLagrangian",
"DeltaLTwoLagrangian", "BLViolatingLagrangian"};

clist = {"Hnorm", "G0norm", "GPnorm", "Wnorm", "Gnorm", "SMEFT$vev",
"SMEFT$MH2", "SMEFT$MW2", "SMEFT$MZ2"};

ylist = {"SMEFT$YL", "SMEFT$YD", "SMEFT$YU"};

slist = {"SMEFTOperatorList", "SMEFTRxiGaugeStatus",
"SMEFTMajoranaNeutrino", "SMEFTCorrect4FermionSign", "SMEFT$WB",
"SMEFT$MB"};

If [ FileExistsQ[ TargetFile ] && (! OptionValue[OverwriteTarget]),
    Print["File " <> TargetFile <> " exists!"];
    TargetExistsQ = AskFunction[ Ask["Overwrite (y/n)?"-> "String"] ][];
];
If[ TargetExistsQ != "y", Print[ "Change name of output file and rerun" ]; Abort[]; ];
Print["Output file will be generated as ",  Style[TargetFile, Bold ] ];

cfile = OpenWrite[TargetFile];

WriteLine[cfile, "(* Generated by SMEFTOutput routine *)"];
date = Date[];
WriteLine[cfile, "(* " <> ToString[date[[3]]] <> "." <>
    ToString[date[[2]]] <> "." <> ToString[date[[1]]] <> " " <>
    ToString[date[[4]]] <> ":" <> ToString[date[[5]]] <> " *)\n"]
WriteLine[cfile, "(* Active operators included in Lagrangian and Feynman rules: " <> ToString[SMEFTOperatorList] <> " *)\n"];

WriteLine[cfile, "(* interaction vertices in mass basis *)\n"];

For[ il=1, il < Length[vlist] + 1,  il++,  
  WriteString[cfile, vlist[[il]] <> " = "];
  Write[cfile, ToExpression[ vlist[[il]] ] ];
  WriteString[cfile, "\n\n"];
];
If[ SMEFTRxiGaugeStatus,
  WriteString[cfile, "GhostVertices = "];
  Write[cfile, GhostVertices];
  WriteString[cfile, "\n\n"];
];

WriteLine[cfile, "(* Lagrangian in mass basis *)\n"];

For[ il=1, il < Length[laglist] + 1,  il++,  
  WriteString[cfile, laglist[[il]] <> " = "];
  Write[cfile, ToExpression[ laglist[[il]] ] ];
  WriteString[cfile, "\n\n"];
];
If[ SMEFTRxiGaugeStatus,
  WriteString[cfile, "GhostLagrangian = "];
  Write[cfile, GhostLagrangian];
  WriteString[cfile, "\n\n"];
];

ZFlocal = {{AZnorm[Index[SU2W, 1], Index[SU2W, 1]], AZnorm[Index[SU2W, 1], Index[SU2W, 2]]},
           {AZnorm[Index[SU2W, 2], Index[SU2W, 1]], AZnorm[Index[SU2W, 2], Index[SU2W, 2]]}};

WriteLine[cfile, "(* corrections to fields and couplings *)\n"];

WriteString[cfile, "AZnorm = "];
Write[cfile, ZFlocal];
WriteString[cfile, "\n\n"];
For[ il=1, il < Length[clist] + 1,  il++,  
  WriteString[cfile, clist[[il]] <> " = "];
  Write[cfile, ToExpression[ clist[[il]] ] ];
  WriteString[cfile, "\n\n"];
];
Clear[i,j];
For[ il=1, il < Length[ylist] + 1,  il++,  
  WriteString[cfile, ylist[[il]] <> "[i,j] = "];
  tmp = OptimizeIndex[ ToExpression[ ylist[[il]] <> "[i,j]" ] ];
  Write[cfile, tmp ];
  WriteString[cfile, "\n\n"];
];

WriteLine[cfile, "(* setup variables *)\n"];

For[ il=1, il < Length[slist] + 1,  il++,  
  WriteString[cfile, slist[[il]] <> " = "];
  Write[cfile, ToExpression[ slist[[il]] ] ];
  WriteString[cfile, "\n\n"];
];

WriteString[cfile, "FeynmanGauge = "];
Write[cfile, SMEFTRxiGaugeStatus ];
WriteString[cfile, "\n\n"];

Close[TargetFile];

]
(* end of SMEFTOutput *)



SMEFTLoadModelFiles = Function[{SMEFTParName},
(* load to FeynRules SMEFT field and parameter definitions *) 
Get[ FileNameJoin[{SMEFT$Path, "definitions", "smeft_fields_WB.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "output", SMEFTParName}] ];
LoadModel[];

(* end of SMEFTLoadModelFiles *)
]




SMEFTLoadLagrangian = Function[{},
(* loads SM and dim6 operator classes *)

(* set gauge *)
SMEFTGaugeRules = If[ ! SMEFTRxiGaugeStatus, {G0|GP|GPbar ->0}, {} ];
(* Load SM Lagrangian *)
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "00_sm.fr"}] ];
(* Load dim6 operators *)
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "01_X3.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "02_phi6.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "03_psi2phi3.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "04_X2phi2.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "05_psi2Xphi.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "06_psi2phi2D.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "07_LlLl.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "08_RrRr.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "09_LlRr.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "10_LrRl.fr"}] ];
Get[ FileNameJoin[{SMEFT$Path, "lagrangian", "11_BL.fr"}] ];

(* end of SMEFTLoadLagrangian *)
]





NumberToString = Function[{x},
(* string display format for floating point numbers *)
Module[{tmp,tmp1,tmp2},

If[ NumberQ[x], 
  If[ Head[x] === Real || Head[x] === Rational, 
    tmp1 = MantissaExponent[N[x]];
    tmp  = ToString[tmp1[[1]]] <> "*^" <> ToString[tmp1[[2]]];
  ];
  If[ Head[x] === Integer, 
    tmp1 = MantissaExponent[x];
    tmp  = ToString[tmp1[[1]]] <> "*^" <> ToString[tmp1[[2]]];
  ];
  If[ Head[x] === Complex, 
    tmp1 = MantissaExponent[N[Re[x]]];
    tmp1 = ToString[tmp1[[1]]] <> "*^" <> ToString[tmp1[[2]]];
    tmp = If[ Im[x] < 0, " - ", " + "]; 
    tmp2 = MantissaExponent[N[Abs[Im[x]]]];
    tmp2 = ToString[tmp2[[1]]] <> "*^" <> ToString[tmp2[[2]]];
    tmp  = tmp1 <> tmp <> tmp2 <> "*I";
  ];  
,
  Print[ "NumberToString function called with argument ",x];
  Print[ "Argument is not a number, correct!"];
  Abort[];
];
  
tmp

]
(* end of NumberToString *)
]



(* functions writing model file entries for scalar and tensor WC's *)

WriteInternalScalarEntry = Function[{cfile,wc},
(* parameter file entry for scalar Warsaw basis WC *)
(* wc argument is member of ScalarWC list *)
Module[{tex,name,pos},

name = ToString[wc];
pos = StringPosition[name, "phi",1];
tex = name;
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];      

WriteLine[cfile, SMEFT$WB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {},"];
WriteLine[cfile, "  InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "  Definitions      -> {" <> SMEFT$WB <> name <> " :> " <> SMEFT$MB <> name <> "},"];
WriteLine[cfile, "  ComplexParameter -> False,"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$WB <> ", " <> tex <> "],"]; 
WriteLine[cfile, "  Description      -> \"Q" <> name <> " dim-6 coupling\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalScalarEntry *)
]



WriteInternalTensor2Entry = Function[{cfile,wc},
(* par file entry for 2-fermion Warsaw basis WC *)
(* wc argument is member of Tensor2WC list *)
Module[{tex,name,pos},

name = ToString[wc[[1]]];
tex = name;
pos = StringPosition[tex, "phi",1];
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];      
pos = StringPosition[tex, "f",1];
If [Length[StringPosition[tex, "f"]]   != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

WriteLine[cfile, SMEFT$WB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "  Definitions      -> {" <> SMEFT$WB <> name <> 
                 "[ff1_,ff2_] :> Module[{ind1,ind2}, " <> 
                 If[ wc[[5]],"Conjugate[",""] <> ToString[wc[[2]]] <> 
                 "[ff1,ind1]" <> If[wc[[5]],"] "," "] <> SMEFT$MB <> name <> 
                 "[ind1,ind2] Conjugate[" <> ToString[wc[[3]]] <> "[ff2,ind2]] ]},"];
WriteLine[cfile, "  Hermitian        -> " <> ToString[ wc[[4]] ] <> ","];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$WB <> ", " <> tex <> "],"]; 
WriteLine[cfile, "  Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalTensor2Entry *)			    
]



WriteInternalTensor4Entry = Function[{cfile,wc},
(* par file entry for 4-fermion Warsaw basis WC *)
(* wc argument is member of Tensor4WC list *)
Module[{name},

name = ToString[wc[[1]]];

WriteLine[cfile, SMEFT$WB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation], Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "  Definitions      -> {" <> SMEFT$WB <> name <> 
                 "[ff1_,ff2_,ff3_,ff4_] :> Module[{ind1,ind2,ind3,ind4}, " <> 
                 If[wc[[6]],"Conjugate[",""] <> ToString[wc[[2]]] <> 
                 "[ff1,ind1]" <> If[wc[[6]],"] "," "] <> 
                 If[wc[[6]],"Conjugate[",""] <> ToString[wc[[3]]] <> 
                 "[ff3,ind3]" <> If[wc[[6]],"] "," "] <> SMEFT$MB <> name <> 
                 "[ind1,ind2,ind3,ind4] Conjugate[" <> ToString[wc[[4]]] <> "[ff2,ind2]] Conjugate[" <>
                 ToString[wc[[5]]] <> "[ff4,ind4]] ]},"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$WB <> ", " <> name <> "],"]; 
WriteLine[cfile, "  Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalTensor4Entry *)			    
]



WriteInternalMassScalarEntry = Function[{cfile,wc},
(* par file entry for scalar mass basis internal WC *)
(* wc argument is member of ScalarWC list *)
Module[{tex,name,pos},

name = ToString[wc];
pos = StringPosition[name, "phi",1];
tex = name;
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];      

WriteLine[cfile, SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {},"];
WriteLine[cfile, "  InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "  ComplexParameter -> False,"];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$MB <> ", " <> tex <> "],"]; 
WriteLine[cfile, "  Description      -> \"Q" <> name <> " dim-6 coupling\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalMassScalarEntry *)
]


WriteInternalMassTensor2Entry = Function[{cfile,wc},
(* par file entry for 2-fermion mass basis internal WC *)
(* wc argument is member of Tensor2WC list *)
Module[{tex,name,pos},

name = ToString[wc[[1]]];
tex = name;
pos = StringPosition[tex, "phi",1];
If [Length[StringPosition[tex, "phi"]] != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];      
pos = StringPosition[tex, "f",1];
If [Length[StringPosition[tex, "f"]]   != 0, tex = StringReplacePart[tex, "\[Phi]", pos] ];

WriteLine[cfile, SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "  Hermitian        -> " <> ToString[ wc[[4]] ] <> ","];
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$MB <> ", " <> tex <> "],"]; 
WriteLine[cfile, "  Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalMassTensor2Entry *)			    
]


WriteInternalMassTensor4Entry = Function[{cfile,wc},
(* wc argument is member of Tensor4WC list *)
(* par file entry for 4-fermion mass basis internal WC *)
Module[{name},

name = ToString[wc[[1]]];

WriteLine[cfile, SMEFT$MB <> name <> " == {"];
WriteLine[cfile, "  ParameterType    -> Internal,"];
WriteLine[cfile, "  Indices          -> {Index[Generation], Index[Generation], Index[Generation], Index[Generation]},"];
WriteLine[cfile, "  InteractionOrder -> {NP,1},"];  
WriteLine[cfile, "  TeX              -> Superscript[" <> SMEFT$MB <> ", " <> name <> "],"]; 
WriteLine[cfile, "  Description      -> \"Q" <> name <> " coupling\""];
WriteLine[cfile, "},\n\n"];

]
(* end of WriteInternalMassTensor4Entry *)			    
]

