(* ::Package:: *)

BeginPackage["MUnitExtras`MUnit`", {"MUnit`"}]


(* ::Section:: *)
(*Public*)


UnevaluatadExpectedOutput::usage =
"\
UnevaluatadExpectedOutput \
is a selector for TestResult that returns unevaluated expected output wrapped \
in HoldForm."


UnevaluatadRawExpectedOutput::usage =
"\
UnevaluatadRawExpectedOutput \
is a selector for TestResult that returns unevaluated expected argument that \
was given to test wrapped in HoldForm."


TestRawInput::usage =
"\
TestRawInput \
is a selector for TestResult that returns unevaluated input argument that was \
given to test wrapped in HoldForm."


TestRawFailureMessage::usage =
"\
TestRawFailureMessage \
is a selector for TestResult that returns raw value of TestFailureMessage \
option from test, in contrast to TestFailureMessage selector that returns \
message generated by TestFailureMessageGenerator."


TestFailureMessageGenerator::usage =
"\
TestFailureMessageGenerator \
is an option to Test that specifies a function used to generate failure \
message that will be logged if the test fails. This function will be called \
with one argument: test result object.

TestFailureMessageGenerator is also a selector for TestResult."


ApplyToInput::usage =
"\
ApplyToInput \
is an option to Test that, if different than None, specifies a function \
applied to input before evaluating it in Test.

ApplyToInput is also a selector for TestResult."


ApplyToExpected::usage =
"\
ApplyToExpected \
is an option to Test that, if different than None, specifies a function \
applied to expected value, given to Test, before evaluating it.

ApplyToExpected is also a selector for TestResult."


InputWrapper::usage =
"\
InputWrapper \
is an option to Test that, if different than None, specifies a function to \
wrap input before evaluating it in Test.

InputWrapper is also a selector for TestResult."


ExpectedWrapper::usage =
"\
ExpectedWrapper \
is an option to Test that, if different than None, specifies a function to \
wrap expected value, given to Test, before evaluating it.

ExpectedWrapper is also a selector for TestResult."


$DefaultTestFailureMessageGenerator::usage =
"\
$DefaultTestFailureMessageGenerator[tr] \
is called to generate final TestFailureMessage for given test result tr."


$FailureMessagesLength::usage =
"\
$FailureMessagesLength \
should be an integer, it denotes maximal number of characters that failure \
message can have."


(* Unprotect all public symbols in this package. *)
Unprotect["`*"];


(* ::Section:: *)
(*Private*)


Begin["`Private`"]


(* ::Subsection:: *)
(*Imports*)


Needs["ProtectionUtilities`"] (* ProtectContextNonVariables *)
Needs["OptionsUtilities`"] (* PrependToOptions *)
Needs["StringUtilities`"] (* StringJoinBy *)


(* ::Subsection:: *)
(*ApplyAndWrapInsideHold*)


ApplyAndWrapInsideHold[
	holdExpr:(_HoldComplete | _Hold | _HoldPattern | _HoldForm),
	apply_, wrap_
] :=
	Switch[{apply, wrap},
		{None, None},
			holdExpr
		,
		{_, None},
			(*
				Since input is wrapped with Hold we apply function at level 1.
			*)
			apply @@@ holdExpr
		,
		{None, _},
			(*
				Since input is wrapped with Hold we map function at holdInput,
				not just wrapp it.
			*)
			wrap /@ holdExpr
		,
		_,
			wrap /@ (apply @@@ holdExpr)
	]


(* ::Subsection:: *)
(*$FailureMessagesLength*)


$FailureMessagesLength = 200


(* ::Subsection:: *)
(*LimitFailureMessageLenght*)


LimitFailureMessageLenght[msg_String] :=
	If[StringLength[msg] > $FailureMessagesLength,
		StringTake[msg, $FailureMessagesLength - 3] <> "..."
	(* else *),
		msg
	]


(* ::Subsection:: *)
(*RemoveSameQWithTrue*)


RemoveSameQWithTrue[HoldForm[SameQ[input_, "True"]]] := HoldForm[input]

RemoveSameQWithTrue[arg_] := arg


(* ::Subsection:: *)
(*DefaultFailureMessageGenerator*)


DefaultFailureMessageGenerator[tr_?TestResultQ] :=
	With[
		{
			failureMessageStr = ToString[TestRawFailureMessage[tr]],
			input = MUnitExtras`Package`MakeString[TestInput[tr]],
			expected =
				MUnitExtras`Package`MakeString[UnevaluatadExpectedOutput[tr]],
			sameTest = SameTest[tr],
			prefix =
				Switch[FailureMode[tr],
					"Failure",
						"Not true that: "
					,
					"MessagesFailure",
						"Incorrect messages generated by: "
					,
					_,
						""
				]
		}
		,
		LimitFailureMessageLenght[
			StringJoinBy[
				failureMessageStr
				,
				prefix <>
					ToString[
						RemoveSameQWithTrue[
							HoldForm[sameTest[input, expected]]
						]
					]
				,
				"Separator" -> " | "
			]
		]
	]


(* ::Subsection:: *)
(*$DefaultTestFailureMessageGenerator*)


$DefaultTestFailureMessageGenerator = DefaultFailureMessageGenerator


(* ::Subsection:: *)
(*$MUnitTestingFunctions*)


$MUnitTestingFunctions = {
	(* MUnit`Test` *)
	Test, TestMatch, TestStringMatch, TestFree, TestStringFree,
	(* MUnit`WRI` *)
	ConditionalTest, ExactTest, ExactTestCaveat, NTest, NTestCaveat, OrTest,
	TestCaveat
}


(* ::Subsection:: *)
(*MUnitVersionedTestOptionsPatch*)


(*	MUnit v1.4 renamed EquivalenceFunction option to SameTest. Add definition
	of Test function that changes option name to one compatible with used
	version of MUnit. *)
With[
	{
		from =
			If[MUnit`Information`$VersionNumber >= 1.4,
				MUnit`EquivalenceFunction
			(* else *),
				SameTest
			]
		,
		to =
			If[MUnit`Information`$VersionNumber >= 1.4,
				SameTest
			(* else *),
				MUnit`EquivalenceFunction
			]
	}
	,
	MUnitVersionedTestOptionsPatch[testFunc_Symbol] := (
		(* Remove option fixing definitions if they exist. *)
		Quiet[
			Scan[
				(testFunc[
					pre:Repeated[_, {2, Infinity}],
					(rule:Rule|RuleDelayed)[#, sameTestValue_],
					post___
				] =.)&
				,
				{SameTest, MUnit`EquivalenceFunction}
			]
			,
			Unset::norep
		];
		
		(* Add proper option fixing definition as first down value. *)
		DownValues[testFunc] =
			Prepend[
				DownValues[testFunc]
				,
				HoldPattern @ testFunc[
					pre:Repeated[_, {2, Infinity}],
					(rule:Rule|RuleDelayed)[from, val_],
					post___
				] :>
					testFunc[pre, rule[to, val], post]
			];
		
		(*	Add up values fixing SetOptions acting on given testFunc.
			If SetOptions gets a list of symbols it's mapped on that list,
			so definitions below are enough. *)
		testFunc /:
			SetOptions[testFunc,
				pre___, (rule: Rule | RuleDelayed)[from, val_], post___
			] :=
				SetOptions[testFunc, pre, rule[to, val], post];
		
		testFunc /:
			SetOptions[testFunc,
				{pre___, (rule: Rule | RuleDelayed)[from, val_], post___}
			] :=
				SetOptions[testFunc, {pre, rule[to, val], post}];
	)
]


(* ::Section:: *)
(*MUnit`Test`Private` modifications*)


Begin["MUnit`Test`Private`"]


AppendTo[$ContextPath, "MUnit`Package`"];
AppendTo[$ContextPath, "MUnitExtras`MUnit`Private`"];


If[!ValueQ[MUnit`Package`$lexicalTestIndex],
	MUnit`Package`$lexicalTestIndex = 0
]


$MUnitTestOptionsSymbols =
	Append[
		$MUnitTestingFunctions
		,
		If[MUnit`Information`$VersionNumber >= 1.4,
			TestResult
		(* else *),
			TestResultObject
		]
	]


PrependToOptions[#,
	TestFailureMessageGenerator -> $DefaultTestFailureMessageGenerator
	,
	(# -> None)& /@
		{ApplyToInput, ApplyToExpected, InputWrapper, ExpectedWrapper}
]& /@
	$MUnitTestOptionsSymbols


(* Fix option incompatibilities beetween MUnit 1.3 and 1.4. *)
MUnitVersionedTestOptionsPatch /@ $MUnitTestOptionsSymbols


DownValues[Test] = DownValues[Test] /.
	If[MUnit`Information`$VersionNumber >= 1.4,
		HoldPattern[Module][
			{moduleArgs__}
			,
			HoldPattern[Catch][
				HoldPattern[CompoundExpression][
					HoldPattern[Block][
						blockArgsList_List
						,
						HoldPattern[CompoundExpression][
							HoldPattern[Block][
								innerBlock1ArgsList_List
								,
								HoldPattern[CompoundExpression][
									HoldPattern[Set][
										{optionVars__},
										HoldPattern[OptionValue][Test, Automatic, {optionValues__}]
									]
									,
									innerCode1__
								]
							]
							,
							code1__
							,
							HoldPattern[Block][
								innerBlock2ArgsList_List
								,
								HoldPattern[With][
									innerWith2ArgsList_List
									,
									HoldPattern[CompoundExpression][
										HoldPattern[Set][
											actualResArg_
											,
											MUnitCheckAll[
												actualOutputSetFunctionArg_[
													actualArg_,
													inputArg_
												]
												,
												optsArg_
												,
												True
											]
										]
										,
										innerCode2__
									]
								]
							]
							,
							code2__
							,
							HoldPattern[Block][
								innerBlock3ArgsList_List
								,
								HoldPattern[With][
									innerWith3ArgsList_List
									,
									HoldPattern[CompoundExpression][
										HoldPattern[Set][
											expectedResArg_
											,
											MUnitCheckAll[
												expectedOutputSetFunctionArg_[
													expectedEvaledArg_,
													expectedArg_
												]
												,
												optsArg_
												,
												True
											]
										]
										,
										innerCode3__
									]
								]
							]
							,
							code3__
						]
					]
					,
					HoldPattern[Module][
						innerModuleArgs_List
						,
						HoldPattern[CompoundExpression][
							HoldPattern[With][
								withArgsList_List
								,
								HoldPattern[CompoundExpression][
									HoldPattern[Set][
										trArg_
										,
										HoldPattern[Sow][
											HoldPattern[newTestResult][
												ntrArgs1:Repeated[_, {3}],
												ntrArgs2:Repeated[Except[_Rule], {5}],
												ntrOpts__
											]
											,
											sowRest__
										]
									]
									,
									code4__
								]
							]
							,
							code5__
						]
					]
				]
				,
				catchRest__
			]
		] :>
			Module[
				{
					holdPreprocessedInput, applyToInput, inputWrapper,
					holdPreprocessedExpected, applyToExpected, expectedWrapper,
					testFailureMessageGenerator
					,
					moduleArgs
				}
				,
				Catch[
					CompoundExpression[
						Block[
							blockArgsList
							,
							CompoundExpression[
								Block[
									innerBlock1ArgsList
									,
									CompoundExpression[
										{
											applyToInput, inputWrapper,
											applyToExpected, expectedWrapper,
											testFailureMessageGenerator, optionVars
										} =
											OptionValue[Test, Automatic, {
												ApplyToInput, InputWrapper,
												ApplyToExpected, ExpectedWrapper,
												TestFailureMessageGenerator,
												optionValues
											}]
										,
										innerCode1
									]
								]
								,
								code1
								,
								Block[
									innerBlock2ArgsList
									,
									With[
										innerWith2ArgsList
										,
										CompoundExpression[
											holdPreprocessedInput =
												ApplyAndWrapInsideHold[
													HoldComplete[inputArg],
													applyToInput,
													inputWrapper
												]
											,
											actualResArg =
												MUnitCheckAll[
													actualOutputSetFunctionArg[
														actualArg,
														ReleaseHold[holdPreprocessedInput]
													]
													,
													optsArg
													,
													True
												]
											,
											innerCode2
										]
									]
								]
								,
								code2
								,
								Block[
									innerBlock3ArgsList
									,
									With[
										innerWith3ArgsList
										,
										CompoundExpression[
											holdPreprocessedExpected =
												ApplyAndWrapInsideHold[
													HoldComplete[expectedArg],
													applyToExpected,
													expectedWrapper
												]
											,
											expectedResArg =
												MUnitCheckAll[
													expectedOutputSetFunctionArg[
														expectedEvaledArg,
														ReleaseHold[holdPreprocessedExpected]
													]
													,
													optsArg
													,
													True
												]
											,
											innerCode3
										]
									]
								]
								,
								code3
							]
						]
						,
						Module[
							innerModuleArgs
							,
							CompoundExpression[
								With[
									withArgsList
									,
									CompoundExpression[
										trArg =
											Sow[
												newTestResult[
													ntrArgs1,
													HoldForm @@ holdPreprocessedInput,
													HoldForm[expectedArg],
													HoldForm @@ holdPreprocessedExpected,
													ntrArgs2,
													ApplyToInput -> applyToInput,
													ApplyToExpected -> applyToExpected,
													InputWrapper -> inputWrapper,
													ExpectedWrapper -> expectedWrapper,
													TestFailureMessageGenerator ->
														testFailureMessageGenerator,
													ntrOpts
												]
												,
												sowRest
											]
										,
										code4
									]
								]
								,
								code5
							]
						]
					]
					,
					catchRest
				]
			]
	(* else *),
		HoldPattern[Module][
			{moduleArgs__}
			,
			HoldPattern[Catch][
				HoldPattern[CompoundExpression][
					HoldPattern[Block][
						blockArgsList_List
						,
						HoldPattern[CompoundExpression][
							HoldPattern[Check][
								HoldPattern[Set][
									{optionVars__},
									HoldPattern[OptionValue][{optionValues__}]
								]
								,
								checkRest__
							]
							,
							code1__
							,
							timeUsedInitExpr:HoldPattern[Set][_, _TimeUsed]
							,
							HoldPattern[Set][
								actualResArg_
								,
								MUnitCheckAll[
									actualOutputSetFunctionArg_[
										actualArg_, inputArg_
									]
									,
									optsArg_
								]
							]
							,
							code2__
							,
							messageListExpr:
								HoldPattern[Set][HoldPattern[$MessageList], {}]
							,
							HoldPattern[Set][
								expectedResArg_
								,
								MUnitCheckAll[
									expectedOutputSetFunctionArg_[
										expectedEvaledArg_,
										expectedArg_
									]
									,
									optsArg_
								]
							]
							,
							code3__
						]
					]
					,
					HoldPattern[Module][
						innerModuleArgs_List
						,
						HoldPattern[CompoundExpression][
							HoldPattern[Set][
								trArg_
								,
								HoldPattern[Sow][
									HoldPattern[newTestResultObject][
										ntrArgs1:Repeated[_, {3}],
										ntrArgs2:Repeated[Except[_Rule], {5}],
										ntrOpts__
									]
									,
									sowRest__
								]
							]
							,
							code4__
						]
					]
				]
				,
				catchRest__
			]
		] :>
			Module[
				{
					holdPreprocessedInput, applyToInput, inputWrapper,
					holdPreprocessedExpected, applyToExpected, expectedWrapper,
					testFailureMessageGenerator
					,
					moduleArgs
				}
				,
				Catch[
					CompoundExpression[
						Block[
							blockArgsList
							,
							CompoundExpression[
								Check[
									{
										applyToInput, inputWrapper,
										applyToExpected, expectedWrapper,
										testFailureMessageGenerator, optionVars
									} =
										OptionValue[{
											ApplyToInput, InputWrapper,
											ApplyToExpected, ExpectedWrapper,
											TestFailureMessageGenerator,
											optionValues
										}]
									,
									checkRest
								]
								,
								code1
								,
								timeUsedInitExpr
								,
								holdPreprocessedInput =
									ApplyAndWrapInsideHold[
										HoldComplete[inputArg],
										applyToInput,
										inputWrapper
									]
								,
								actualResArg =
									MUnitCheckAll[
										actualOutputSetFunctionArg[
											actualArg,
											ReleaseHold[holdPreprocessedInput]
										],
										optsArg
									]
								,
								code2
								,
								messageListExpr
								,
								holdPreprocessedExpected =
									ApplyAndWrapInsideHold[
										HoldComplete[expectedArg],
										applyToExpected,
										expectedWrapper
									]
								,
								expectedResArg =
									MUnitCheckAll[
										expectedOutputSetFunctionArg[
											expectedEvaledArg,
											ReleaseHold[holdPreprocessedExpected]
										]
										,
										optsArg
									]
								,
								code3
							]
						]
						,
						Module[
							innerModuleArgs
							,
							CompoundExpression[
								trArg = Sow[
									newTestResultObject[
										ntrArgs1,
										HoldForm @@ holdPreprocessedInput,
										HoldForm[expectedArg],
										HoldForm @@ holdPreprocessedExpected,
										ntrArgs2,
										ApplyToInput -> applyToInput,
										ApplyToExpected -> applyToExpected,
										InputWrapper -> inputWrapper,
										ExpectedWrapper -> expectedWrapper,
										TestFailureMessageGenerator ->
											testFailureMessageGenerator,
										ntrOpts
									]
									,
									sowRest
								]
								,
								code4
							]
						]
					]
					,
					catchRest
				]
			]
	]


With[
	{
		newTestResultFunction =
			If[MUnit`Information`$VersionNumber >= 1.4,
				newTestResult
			(* else *),
				newTestResultObject
			]
	}
	,
	DownValues[testError] = DownValues[testError] /.
		HoldPattern[Module][
			{testIDVar_, testErrorActionVar_, trVar_}
			,
			HoldPattern[CompoundExpression][
				code1__
				,
				HoldPattern[Set][
					trVar_
					,
					HoldPattern[Sow][
						HoldPattern[newTestResultFunction][
							arg1_,
							arg2_,
							args:Repeated[_, {6}],
							opts__
						]
						,
						sowRest___
					]
				]
				,
				code2__
			]
		] :>
			Module[
				{testIDVar, testErrorActionVar, trVar}
				,
				CompoundExpression[
					code1
					,
					trVar = Sow[
						newTestResultFunction[
							arg1,
							arg2,
							arg1,
							arg1,
							arg1,
							args,
							opts,
							TestErrorAction -> testErrorActionVar
						]
						,
						sowRest
					]
					,
					code2
				]
			]
]


If[MUnit`Information`$VersionNumber >= 1.4,
	DownValues[newTestResult] = DownValues[newTestResult] /.
		RuleDelayed[
			HoldPattern[HoldPattern][
				HoldPattern[newTestResult][
					testPatt_,
					failureModePatt_,
					inputPatt_,
					restPatts:Repeated[_, {5}],
					optsPatt:HoldPattern[Pattern][
						optsArg_,
						HoldPattern[OptionsPattern][]
					]
				]
			]
			,
			HoldPattern[Module][
				outerModuleArgs_List
				,
				HoldPattern[With][
					withArgs_List
					,
					HoldPattern[Module][
						moduleArgs_List
						,
						HoldPattern[CompoundExpression][
							code1__
							,
							HoldPattern[TagSetDelayed][_, _TestFailureMessage, _]
							,
							code2__
							,
							objArg_
						]
					]
				]
			]
		] :>
			RuleDelayed[
				HoldPattern[
					newTestResult[
						testPatt,
						failureModePatt,
						rawInput_,
						inputPatt,
						unevaluatedRawExpectedOutput_,
						unevaluatedExpectedOutput_,
						restPatts,
						optsPatt
					]
				]
				,
				Module[
					outerModuleArgs
					,
					With[
						withArgs
						,
						Module[
							moduleArgs
							,
							CompoundExpression[
								code1
								,
								objArg /: TestRawFailureMessage[objArg] :=
									OptionValue[TestResult, {optsArg}, TestFailureMessage];
								,
								objArg /: TestFailureMessage[objArg] :=
									TestFailureMessageGenerator[objArg][objArg];
								,
								code2
								,
								objArg /: TestRawInput[objArg] :=
									rawInput,
								objArg /: UnevaluatadRawExpectedOutput[objArg] :=
									unevaluatedRawExpectedOutput,
								objArg /: UnevaluatadExpectedOutput[objArg] :=
									unevaluatedExpectedOutput,
								objArg /: ApplyToInput[objArg] :=
									OptionValue[TestResult, {optsArg}, ApplyToInput],
								objArg /: ApplyToExpected[objArg] :=
									OptionValue[TestResult, {optsArg}, ApplyToExpected],
								objArg /: InputWrapper[objArg] :=
									OptionValue[TestResult, {optsArg}, InputWrapper],
								objArg /: ExpectedWrapper[objArg] :=
									OptionValue[TestResult, {optsArg}, ExpectedWrapper],
								objArg /: TestFailureMessageGenerator[objArg] :=
									OptionValue[TestResult, {optsArg}, TestFailureMessageGenerator]
								,
								objArg /: MUnit`EquivalenceFunction[objArg] :=
									OptionValue[TestResult, {optsArg}, SameTest]
								,
								objArg
							]
						]
					]
				]
			]
(* else *),
	DownValues[newTestResultObject] = DownValues[newTestResultObject] /.
		RuleDelayed[
			HoldPattern[HoldPattern][
				HoldPattern[newTestResultObject][
					testPatt_,
					failureModePatt_,
					inputPatt_,
					restPatts:Repeated[_, {5}],
					optsPatt:HoldPattern[Pattern][
						optsArg_,
						HoldPattern[OptionsPattern][]
					]
				]
			]
			,
			HoldPattern[With][
				withArgs_List
				,
				HoldPattern[Module][
					moduleArgs_List
					,
					HoldPattern[CompoundExpression][
						code1__
						,
						HoldPattern[TagSetDelayed][_, _TestFailureMessage, _]
						,
						code2__
						,
						objArg_
					]
				]
			]
		] :>
			RuleDelayed[
				HoldPattern[
					newTestResultObject[
						testPatt,
						failureModePatt,
						rawInput_,
						inputPatt,
						unevaluatedRawExpectedOutput_,
						unevaluatedExpectedOutput_,
						restPatts,
						optsPatt
					]
				]
				,
				With[
					withArgs
					,
					Module[
						moduleArgs
						,
						CompoundExpression[
							code1
							,
							objArg /: TestRawFailureMessage[objArg] :=
								OptionValue[TestResultObject, {optsArg}, TestFailureMessage];
							,
							objArg /: TestFailureMessage[objArg] :=
								TestFailureMessageGenerator[objArg][objArg];
							,
							code2
							,
							objArg /: TestRawInput[objArg] :=
								rawInput,
							objArg /: UnevaluatadRawExpectedOutput[objArg] :=
								unevaluatedRawExpectedOutput,
							objArg /: UnevaluatadExpectedOutput[objArg] :=
								unevaluatedExpectedOutput,
							objArg /: ApplyToInput[objArg] :=
								OptionValue[TestResultObject, {optsArg}, ApplyToInput],
							objArg /: ApplyToExpected[objArg] :=
								OptionValue[TestResultObject, {optsArg}, ApplyToExpected],
							objArg /: InputWrapper[objArg] :=
								OptionValue[TestResultObject, {optsArg}, InputWrapper],
							objArg /: ExpectedWrapper[objArg] :=
								OptionValue[TestResultObject, {optsArg}, ExpectedWrapper],
							objArg /: TestFailureMessageGenerator[objArg] :=
								OptionValue[TestResultObject, {optsArg}, TestFailureMessageGenerator]
							,
							objArg /: SameTest[objArg] :=
								OptionValue[TestResultObject, {optsArg}, MUnit`EquivalenceFunction]
							,
							objArg
						]
					]
				]
			]
	]


End[]


End[]


(* ::Section:: *)
(*Public symbols protection*)


ProtectContextNonVariables[];


EndPackage[]
