(* ::Package:: *)

BeginPackage["OptionsUtilities`"]


(* ::Section:: *)
(*Usage messages*)


DefineRequiredOptions::usage =
"\
DefineRequiredOptions[symbol, opt1Name, opt2Name, ...] or \
DefineRequiredOptions[symbol, {opt1Name, opt2Name, ...}] \
defines, for symbol, options opt1Name, opt2Name, ... such that, if those \
options are used without setting them explicitly, a warning message is \
printed.\

DefineRequiredOptions[{sym1, sym2, ...}, opt1Name, opt2Name, ...] or \
DefineRequiredOptions[{sym1, sym2, ...}, {opt1Name, opt2Name, ...}] \
defines, for symbols sym1, sym2, ..., options opt1Name, opt2Name, ... such \
that, if those options are used without setting them explicitly, a warning \
message is printed."


DeleteOptionDuplicates::usage =
"\
DeleteOptionDuplicates[opts] \
returns list of unique options from opts. If opts contains more than one \
option with same name first one is in returned list."


DelegateOptions::usage =
"\
DelegateOptions[explicitOpts, defaultFromFunc, toFunc] \
returns list of options for toFunc with values taken from given \
explicitOpts If option is not present in explicitOpts it's value is taken \
from default options of defaultFromFunc."


PrependToOptions::usage =
"\
PrependToOptions[func, opts] or PrependToOptions[func, {opts}] \
prepends opts to list of possible options of func."


CopyFeatures::usage =
"\
CopyFeatures[fromFunc, toFunc] \
copies attributes and options of fromFunc to toFunc.\

CopyFeatures[{fromFunc1, fromFunc2, ...}, toFunc] \
copies attributes and options of all fromFunci to toFunc.\

CopyFeatures[fromFunc, {toFunc1, toFunc2, ...}] \
copies attributes and options of fromFunc to each of toFunci.\

CopyFeatures[{fromFunc1, fromFunc2, ...}, {toFunc1, toFunc2, ...}] \
copies attributes and options of all fromFunci to each of toFunci."


SaveOptions::usage =
"\
SaveOptions[symbol1, symbol2, ...] or \
SaveOptions[{symbol1, symbol2, ...}] \
saves options of symbol1, symbol2, ... and returns unique identifier that can \
be used to restore saved options."


RestoreOptions::usage =
"\
RestoreOptions[] \
restores last saved options.\

RestoreOptions[id] \
restores options saved with given id.\

RestoreOptions[{sym1, sym2, ...}] \
restores last saved options for symbols sym1, sym2, ...\

RestoreOptions[id, {sym1, sym2, ...}] \
restores options for symbols sym1, sym2, ... saved with given id."


(* ::Section:: *)
(*Implementation*)


(*
	Unprotect all symbols in this context
	(all public symbols provided by this package)
*)
Unprotect["`*"]


Begin["`Private`"]


(* ::Subsection:: *)
(*General messages*)


General::setOption = "Set `1` option to use `2` function.";


General::unknownOptionValue =
"`1` is not valid value for option `2`. Valid values are: `3`. \
Message generated by `4`."


(* ::Subsection:: *)
(*DefineRequiredOptions*)


DefineRequiredOptions[sym_Symbol, options:{(_Symbol | _String)..}] := (
	Options[sym] = Join[
		Options[sym],
		(# :> Message[sym::setOption, #, sym])& /@ options
	]
)


DefineRequiredOptions[sym_Symbol, options:(_Symbol | _String)..] :=
	DefineRequiredOptions[sym, {options}]


DefineRequiredOptions[
	symbols:{_Symbol..},
	options:(((_Symbol | _String)..) | {(_Symbol | _String)..})
] :=
	DefineRequiredOptions[#, {options}]& /@ symbols


(* ::Subsection:: *)
(*DeleteOptionDuplicates*)


DeleteOptionDuplicates[opts___?OptionQ] :=
	DeleteDuplicates[Flatten[{opts}], SameQ @@ First /@ {##}&]


(* ::Subsection:: *)
(*DelegateOptions*)


DelegateOptions[
	explicitOpts___?OptionQ,
	defaultFromFunc_Symbol,
	toFunc:(_Symbol | _String)
] :=
	DeleteOptionDuplicates[
		FilterRules[
			Flatten[{explicitOpts, Options[defaultFromFunc]}],
			Options[toFunc]
		]
	]


(* ::Subsection:: *)
(*PrependToOptions*)


Options[PrependToOptions] = {"DeleteOptionDuplicates" -> True}


PrependToOptions[
	func:(_Symbol | _String),
	opts:{__?OptionQ},
	OptionsPattern[]
] :=
	Options[func] =
		If[OptionValue["DeleteOptionDuplicates"],
			DeleteOptionDuplicates
		(* else *),
			Join
		][opts, Options[func]]

PrependToOptions[func:(_Symbol | _String), opts__?OptionQ] :=
	PrependToOptions[func, {opts}]


(* ::Subsection:: *)
(*CopyFeatures*)


Options[CopyFeatures] = {
	"ExcludeAttributes" -> {Protected},
	"ExcludeOptions" -> {},
	"DeleteOptionDuplicates" -> True
}


CopyFeatures[fromFuncs:{__Symbol}, toFuncs:{__Symbol}, OptionsPattern[]] :=
	Module[
		{
			options =
				FilterRules[
					Join @@ Options /@ fromFuncs,
					Except[OptionValue["ExcludeOptions"]]
				]
			,
			attributes =
				DeleteCases[
					Join @@ Attributes /@ fromFuncs,
					Alternatives @@ OptionValue["ExcludeAttributes"]
				]
		}
		,
		If[OptionValue["DeleteOptionDuplicates"],
			options = DeleteOptionDuplicates[options];
		];
		
		(
			Options[#] = options;
			Attributes[#] = attributes;
		)& /@ toFuncs;
	]

CopyFeatures[fromFunc_Symbol, toFuncs:{__Symbol}, opts:OptionsPattern[]] :=
	CopyFeatures[{fromFunc}, toFuncs, opts]


CopyFeatures[fromFuncs:{__Symbol}, toFunc_Symbol, opts:OptionsPattern[]] :=
	CopyFeatures[fromFuncs, {toFunc}, opts]


CopyFeatures[fromFunc_Symbol, toFuncs_Symbol, opts:OptionsPattern[]] :=
	CopyFeatures[{fromFunc}, {toFuncs}, opts]


(* ::Subsection:: *)
(*Save and Restore Options*)


$SavedOptions = {}


(* ::Subsubsection:: *)
(*SaveOptions*)


Options[SaveOptions] = {
	"UniqueIdentifier" :> Unique["savedOptions$"],
	"SaveEmptyOptions" -> False
}


SaveOptions[symbols:{__Symbol}, OptionsPattern[]] :=
	Module[
		{
			id = OptionValue["UniqueIdentifier"],
			optToSave = Sort[(# -> Options[#])& /@ symbols]
		}
		,
		If[!OptionValue["SaveEmptyOptions"],
			optToSave = DeleteCases[optToSave, _ -> {}]
		];
		
		If[MemberQ[$SavedOptions, id -> _],
			optToSave =
				Join[
					FilterRules[
						id /. $SavedOptions,
						Except[Alternatives @@ symbols]
					]
					,
					optToSave
				];
			
			$SavedOptions = DeleteCases[$SavedOptions, id -> _]
		];
		
		AppendTo[
			$SavedOptions,
			id -> (MapAt[DeleteOptionDuplicates, #, -1]& /@ optToSave)
		];
		
		id
	]


SaveOptions[symbols__Symbol, opt:OptionsPattern[]] :=
	SaveOptions[{symbols}, opt]


(* ::Subsubsection:: *)
(*RestoreOptions*)


RestoreOptions::unknownId =
"Given id: `1` is not available, \
available identifiers for saved options are: `2`. \
Message generated by `3`."

RestoreOptions::unknownSymbols =
"Not all given symbols: `1` have saved options, \
symbols with saved options with used identifier: `2` are: `3`. \
Missing symbols are `4`. \
Message generated by `5`."

RestoreOptions::noSavedOptions =
"There are no saved options to restore. \
Message generated by `1`."


Options[RestoreOptions] = {"DeleteUsedId" -> Automatic}


functionCall:RestoreOptions[
	idArg_Symbol:Automatic,
	symbols:({__Symbol} | All):All,
	OptionsPattern[]
] := (
	If[Length[$SavedOptions] < 1,
		Message[
			RestoreOptions::noSavedOptions,
			HoldForm[functionCall]
		];
		Return[$Failed]
	];
	
	Module[
		{
			savedOptionsRule,
			id,
			savedOptions,
			savedOptionsForSymbols,
			savedSymbols
		}
		,
		
		If[idArg === Automatic,
			savedOptionsRule = Last[$SavedOptions];
			id = First[savedOptionsRule];
			savedOptions = Last[savedOptionsRule];
		(* else *),
			If[!MemberQ[$SavedOptions, idArg -> _],
				Message[
					RestoreOptions::unknownId,
					idArg,
					First /@ $SavedOptions,
					HoldForm[functionCall]
				];
				Return[$Failed]
			];
			
			id = idArg;
			savedOptions = id /. $SavedOptions;
		];
		
		If[symbols === All,
			savedOptionsForSymbols = savedOptions;
		(* else *),
			savedSymbols = First /@ savedOptions;
			
			If[Complement[symbols, savedSymbols] =!= {},
				Message[
					RestoreOptions::unknownSymbols,
					symbols,
					id,
					savedSymbols,
					Complement[symbols, savedSymbols],
					HoldForm[functionCall]
				];
				Return[$Failed]
			];
			
			savedOptionsForSymbols = FilterRules[savedOptions, symbols];
		];
		
		Switch[OptionValue["DeleteUsedId"],
			Automatic,
				If[symbols === All,
					$SavedOptions = DeleteCases[$SavedOptions, id -> _];
				];
			,
			True,
				$SavedOptions = DeleteCases[$SavedOptions, id -> _];
			,
			False,
				Null;
			,
			_,
				Message[
					RestoreOptions::unknownOptionValue,
					OptionValue["DeleteUsedId"],
					"DeleteUsedId",
					{Automatic, True, False},
					HoldForm[functionCall]
				];
				Return[$Failed]
		];
		
		SetOptions @@@ savedOptionsForSymbols;
	]
)


End[]


(* ::Subsection:: *)
(*Public symbols protection*)


(*
	Protect all symbols in this context
	(all public symbols provided by this package)
*)
Protect["`*"]


EndPackage[]
