subroutine level(qv,qj,isot,nonrel,N,ran,na_cutoff,outp)
!Returns the energy contributions and the total energy
!for a given rovibrational level.
!
!References point to J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019),
!unless stated otherwise.
!
!qv   -- vibrational quantum number (integer, intent(in))
!
!qj   -- rotational quantum number (integer, intent(in))
!
!isot -- hydrogen molecule isotopologue (character(len=2), intent(in)), one of:
!        "H2"/"D2"/"T2"/"HD"/"HT"/"DT"/"PC" (the last one prints the physical
!        constants used)
!
!nonrel -- an evaluation scheme (character(len=1), intent(in)), one of:
!          "A" -- both outp%E2full and outp%E2napt are evaluated, and outp%E2=outp%E2full
!                 if only available. If the fully nonadiabatic energy is unavailable, 
!                 then outp%E2full=NaN and outp%E2=outp%E2napt. The most fail-safe choice.
!          "N" -- only outp%E2napt is evaluated, outp%E2full=NaN and outp%E2=outp%E2napt.
!          "B" -- Same as "N", but outp%E2napt and outp%E2 are  just in BO approximation 
!                 (very unaccurate, not recommended!).
!          "F" -- only outp%E2full is evaluated and outp%E2=outp%E2full.
!                 If the fully nonadiabatic energy is unavailable, outp%E2full=NaN
!                 and outp%E2=NaN.
!
!N     -- number of DVR grid points, confer Section III C (integer, intent(in));
!
!ran  -- DVR grid range, R_N in Eq. 44  (double precision, intent(in));
!
!na_cutoff -- the heteronuclear nonadiabatic potential dEnaprim(R) (Eq. 37) is divergent at R=0;
!             if the DVR step is too short, it can cause problems; this parameter sets this potential
!             to dEnaprim(R)=dEnaprim(na_cutoff) for R<na_cutoff;
!
!outp -- container variable for the energy contributions (confer Eq. 1),
!        defined in "h2spectr_types.f90" and
!        described in detail in the paper (type(output), intent(out)):
!        outp%E2full -- the nonrelativistic energy E(2) from four-body calculations (Section III A)
!        outp%E2napt -- the nonrelativistic energy E(2) in the NAPT approach (Section III B)
!        outp%E2     -- one of the above, depending how "nonrel" is set
!        outp%Ena    -- the nonadiabatic E(2,2) correction, used to estimate the NAPT error
!        outp%E4bo   -- the relativistic BO correction E(4,0) (Eq. 61)
!        outp%E4na   -- the first-order part of the relativistic nonadiabatic correction E(4,1) (first line of Eq. 68)
!        outp%E4sec  -- the 2nd-order part of E(4,1) (second line of Eq. 68)
!        outp%E4     -- the total relativistic energy E(4):outp%E4bo+outp%E4na+outp%E4sec
!        outp%E5     -- the leading QED correction (Eq. 69)
!        outp%E6bo   -- the first-order part of the HQED E(6,0) BO correction (first line of Eq. 74)
!        outp%E6sec  -- the 2nd-order part of E(6,0) (second line of Eq. 74)
!        outp%E6     -- the total HQED correction E(6,0): outp%E6bo+outp%E6sec
!        outp%E7     -- the HQED E(7) correction (Eq. 75)
!        outp%Efs    -- the finite-nucleus-size E(4)_FS correction (Eq. 77)
!        outp%Etot   -- the total energy of the rovibrational level
!       
!        outp%errE2    -- the nonrelativistic E(2) energy uncertainty
!        outp%errE4    -- the total relativistic E(4) energy uncertainty
!        outp%errE5    -- the leading QED E(5,0) correction uncertainty
!        outp%errE6    -- the HQED E(6,0) correction uncertainty
!        outp%errE7    -- the HQED E(7) correction uncertainty
!        outp%errEfs   -- the finite-nucleus-size E(4)_FS correction uncertainty due to next neglected NAPT term
!        outp%rerr     -- the uncertainty due to finite accuracy of the nuclear charge radii
!        outp%errEtot  -- the total energy uncertainty
!
!        outp%E2which  -- indicated outp%E2 calculation type ("FULL"/"NAPT"/"BO  "/"NAN ")
!        outp%pm       -- the reduced nuclear mass in use
!
!        outp%wav      -- value of the nuclear radial wave function on the grid points
!       
!        outp%error_switch -- error estimation method for transitions: 
!                              default -- for each contribution take the bigger error from the pair
!                                         of levels taking part in the transition
!                              1       -- proportional estimation: it is the same method as for levels;
!                                         it can lead to lesser error bars and is preferable for 
!                                         close-lying-state transitions like (0,1)->(0,0), but can fail if the
!                                         radial functions differ much between the two states
!                                                                             
!        outp%HF??     -- the hyperfine parameters, valid for H2, D2, HD and HT molecules
!
!        All of the above components are double precision, except outp%E2which (character(len=4)) and
!        error_switch (integer).
!


use h2spectr_types
use control_parameters
USE,INTRINSIC :: IEEE_ARITHMETIC
      IMPLICIT NONE
      integer,intent(in)::qv,qj
      character(len=2),intent(in)::isot
      character(len=1),intent(in)::nonrel      
      integer,intent(in) ::N
      double precision, intent(in) ::ran,na_cutoff
      type(output),intent(out)::outp

      type(potentials)::p
      integer IL,IU,I,J,K,INFO,IE,L,vstate
      integer,allocatable:: IWORK(:),IFAIL(:),IPIV(:)
      double precision,allocatable:: WORK(:),W(:),Z(:,:),Zn(:,:),Wn(:),H(:)
      double precision VL,VU,ABSTOL,DLAMCH,pm,u,alpha,one,lambda,pi
      double precision mprot,mdeut,ma,mtrit,ns,bhr,hrad,drad,trad,dr
      double precision hrerr,drerr,trerr
      logical :: nonad_present
      double precision :: nonad_E2,ma6sec,ma4sec
      double precision, allocatable::chi(:)
      double precision :: tmp1,tmp2,ma4,ma4na,ma5,ma6,deltaia,ma7,fs
      character(len=30) :: infile

      parameter (pi=3.14159265358979324d0)
      parameter (hrad=0.8414d0) !Proton charge radius [1]
      parameter (hrerr=1.9d-3) !and its uncertainty
      parameter (drad=2.12799d0) !Deuteron charge radius [1]
      parameter (drerr=7.4d-4) !and its uncertainty
      parameter (trad=1.7591d0)  !Triton charge radius [2]
      parameter (trerr=3.63d-2) !and its uncertainty
      parameter (mprot = 1836.15267343D0) !Proton mass [1]
      parameter (mdeut = 3670.48296788D0) !Deuteron mass [1]
      parameter (mtrit = 5496.92153573D0) !Triton mass [1]
      parameter (alpha = 7.2973525693d-3) !Fine structure constant [1]
      parameter (u = 219474.63136320d0) !2*Rydberg constant [1] -- to convert to cm-1 (includes alpha**2)
      parameter (bhr=0.529177210903d5) !fm to Bohr conversion [1]
      parameter (one = 1.d0)

      ![1] 2018 CODATA recommended values, https://physics.nist.gov/cuu/Constants
      ![2] I. Angeli, K.P. Marinova, At. Data Nucl. Data Tables 99, 69 (2013)


      lambda=0.d0 !Heteronuclear-unique factor, Eq. 36
      if((nonrel.ne."N").and.(nonrel.ne."B").and.(nonrel.ne."A").and.(nonrel.ne."F")) then
        write(*,*)"Unrecognised nonrel switch: ",nonrel,"! It should be one of N/B/A/L."
        stop
      endif

      !Set isotopologue-specific variables
      select case(isot)
      case("H2")
          pm=mprot/2 !reduced mass
          ma=(mprot+1)/2 !reduced mass of a system of 2 atoms
          ns=(hrad/bhr)**2 !nuclear size
          outp%rerr=2*hrerr/hrad !the uncertainty due to finite accuracy of the nuclear charge radii

      case("D2")
          pm=mdeut/2
          ma=(mdeut+1)/2
          ns=(drad/bhr)**2
          outp%rerr=2*drerr/drad

      case("T2")
          pm=mtrit/2
          ma=(mtrit+1)/2
          ns=(trad/bhr)**2
          outp%rerr=2*trerr/trad

      case("HD")
          pm=1/(1/mprot+1/mdeut)
          ma=1/(1/(mprot+1)+1/(mdeut+1))
          ns=((hrad/bhr)**2+(drad/bhr)**2)/2
          lambda=-(1/mdeut-1/mprot)/2 !Heteronuclear-unique factor, Eq. 36
         !The values of the proton and deuteron radii are almost 100% correlated,
         !hence a different formula for this error
          outp%rerr=2*sqrt((2*hrad*hrerr)**2)/(hrad**2+drad**2) 

      case("HT")
          pm=1/(1/mprot+1/mtrit)
          ma=1/(1/(mprot+1)+1/(mtrit+1))
          ns=((hrad/bhr)**2+(trad/bhr)**2)/2
          lambda=-(1/mtrit-1/mprot)/2
          outp%rerr=sqrt((2*hrad*hrerr)**2+(2*trad*trerr)**2)/(hrad**2+trad**2)

      case("DT")
          pm=1/(1/mdeut+1/mtrit)
          ma=1/(1/(mdeut+1)+1/(mtrit+1))
          ns=((drad/bhr)**2+(trad/bhr)**2)/2
          lambda=-(1/mtrit-1/mdeut)/2
          outp%rerr=sqrt((2*drad*drerr)**2+(2*trad*trerr)**2)/(drad**2+trad**2)

      case("PC") !Prints used constants, together with uncertainties and references
        write(*,'(a,7x,f7.4,a7)') "Proton charge radius [1]   ",hrad,"(19) fm"
        write(*,'(a,8x,f7.5,a7)') "Deuteron charge radius [1] ",drad,"(74) fm"
        write(*,'(a,8x,f6.4,a8)') "Triton charge radius [2]   ",trad,"(363) fm"
        write(*,'(a,4x,f14.8,a4)') "Proton mass [1]            ",mprot,"(11)"
        write(*,'(a,4x,f14.8,a4)') "Deuteron mass [1]          ",mdeut,"(13)"
        write(*,'(a,4x,f14.8,a4)') "Triton mass [1]            ",mtrit,"(27)"
        write(*,'(a,2x,f16.6,a8)') "Rydberg constant [1]     ",100*u/2,"(21) m-1"
        write(*,'(a,16x,f14.10,a7)') "Bohr radius [1]  ",bhr/1000,"(80) pm"
        write(*,'(a,2x,f18.10,a9)') "Fine structure constant [1]",alpha*1000,"(11) E-03"
        write(*,*)
        write(*,'(a)') "[1] 2018 CODATA recommended values, https://physics.nist.gov/cuu/Constants"
        write(*,'(a)') "[2] I. Angeli, K.P. Marinova, At. Data Nucl. Data Tables 99, 69 (2013)"
        write(*,*)"--------"
        write(*,'(a)')'It should not be significant for the overall quality of the results, but the fully' 
        write(*,'(a)')'nonadiabatic nonrelativistic results were calculated with the old nuclear masses,'
        write(*,'(a)')'from: P. J. Mohr, D. B. Newell, and B. N. Taylor, Rev. Mod. Phys. 88, 035009 (2016):'
        write(*,'(a,4x,f14.8,a4)') "Proton mass                ",1836.15267389d0,"(17)"
        write(*,'(a,4x,f14.8,a4)') "Deuteron mass              ",3670.48296785d0,"(13)"
        write(*,'(a,4x,f14.8,a4)') "Triton mass                ",5496.92153588d0,"(26)"

        stop
      
      case default 
        write(*,*)"Unrecognised isotopologue!"
        stop
      end select

      !Set the grid separation (Eq. 44)
      dr=ran/N

      !Allocate arrays
      allocate(IWORK(5*N),IFAIL(N),IPIV(N))
      allocate(WORK(8*N),W(N),Z(N,N),Zn(N,N),Wn(N),H(N*(N+1)/2),chi(N))
      allocate(outp%wav(N,2))
      call allocpot(p,N)
      


      outp%pm=pm
      !Prepare DVR representations of the correction potentials and store them in "p"
      call loader(p,lambda,alpha,dr,N,na_cutoff)

       vstate=qv+1
       L=qj
       ABSTOL =  2*DLAMCH('S') !Convergence criterion = twice the underflow threshold

!----------------------------------------
!DVR&nonrelativistic section

!Initialise with NaNs
outp%E2=IEEE_VALUE(outp%E2,IEEE_QUIET_NAN)
outp%E2full=IEEE_VALUE(outp%E2full,IEEE_QUIET_NAN)
outp%E2napt=IEEE_VALUE(outp%E2napt,IEEE_QUIET_NAN)
outp%E2which="NAN "


      !DVR CALCULATION: BO, confer Eqs. 51 and 52
       DO I=1,N
        DO J=1,I
         K = I*(I-1)/2+J
         IF(I.EQ.J) THEN
          H(K) = one/(2*pm*dr**2)*(pi**2/3-one/(2*I**2)) &
               + L*(L+1)/(2*pm*p%R(I)**2)+p%V(I)

         ELSE
         H(K) = (-1)**MOD(I-J,2)/(pm*dr**2)*(one/(I-J)**2-one/(I+J)**2)
         ENDIF
        ENDDO
       ENDDO

        CALL DSPEVX('V','A','U',N,H,VL,VU,IL,IU,ABSTOL,IE,W,Z,N, WORK,IWORK,IFAIL,INFO)

      Chi=Z(:,vstate) !BO radial function in a DVR form (left-hand side of Eq. 47)
      outp%wav(:,2)=Chi/sqrt(dr) !Values of the radial function on the grid points
      outp%wav(:,1)=p%R !The grid points themselves
!----------------------------------------
! Hyperfine parameters
! cp, cd, ct, d1, d2 are the hyperfine parameters in Ramsey's notation; 
! qd is the electric field gradient at the position of deuteron
! For details see Komasa, Puchalski, Pachucki, PRA 102, 012814 (2020)
      if(HF_bool) then
        print '(2x,a)','Hyperfine parameters   Value       Error'
        select case (isot)
          case ('HD')
            outp%HFcp=0.d0
            outp%HFcd=0.d0
            outp%HFd1=0.d0
            !outp%HFqd=0.d0
            outp%HFd2=0.d0
            do i=1,N
              outp%HFcp=outp%HFcp+chi(i)*p%HFcp(i)*chi(i)
              outp%HFcd=outp%HFcd+chi(i)*p%HFcd(i)*chi(i)
              outp%HFd1=outp%HFd1+chi(i)*p%HFd1(i)*chi(i)
              !outp%HFqd=outp%HFqd+chi(i)*p%HFqd(i)*chi(i)
              outp%HFd2=outp%HFd2+chi(i)*p%HFd2(i)*chi(i)
            enddo
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cp > ",outp%HFcp,abs(outp%HFcp/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cd > ",outp%HFcd,abs(outp%HFcd/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d1 > ",outp%HFd1,abs(outp%HFd1/pm)
            !print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< qD > ",outp%HFqd,abs(outp%HFqd/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d2 > ",outp%HFd2,abs(outp%HFd2/pm)
          case ('HT')
            outp%HFcp=0.d0
            outp%HFcd=0.d0  ! cd is used to store also the HT data
            outp%HFd1=0.d0
            do i=1,N
              outp%HFcp=outp%HFcp+chi(i)*p%HFcp(i)*chi(i)
              outp%HFcd=outp%HFcd+chi(i)*p%HFcd(i)*chi(i)
              outp%HFd1=outp%HFd1+chi(i)*p%HFd1(i)*chi(i)
            enddo
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cp > ",outp%HFcp,abs(outp%HFcp/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< ct > ",outp%HFcd,abs(outp%HFcd/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d1 > ",outp%HFd1,abs(outp%HFd1/pm)
          case ('H2')
            outp%HFcp=0.d0
            outp%HFd1=0.d0
            do i=1,N
              outp%HFcp=outp%HFcp+chi(i)*p%HFcp(i)*chi(i)
              outp%HFd1=outp%HFd1+chi(i)*p%HFd1(i)*chi(i)
            enddo
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cp > ",outp%HFcp,abs(outp%HFcp/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d1 > ",outp%HFd1,abs(outp%HFd1/pm)
          case ('D2')
            outp%HFcd=0.d0
            outp%HFd1=0.d0
            do i=1,N
              outp%HFcd=outp%HFcd+chi(i)*p%HFcd(i)*chi(i)
              outp%HFd1=outp%HFd1+chi(i)*p%HFd1(i)*chi(i)
            enddo
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cd > ",outp%HFcd,abs(outp%HFcd/pm)
            print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d1 > ",outp%HFd1,abs(outp%HFd1/pm)
        end select
        print '(1x,23a2)',('- ',i=1,23)
      endif
!----------------------------------------
!      if(HF_bool .and. (isot=='HD' .or. isot=='H2' .or. isot=='D2' .or. isot=='HT')) then
!        outp%HFcp=0.d0
!        outp%HFcd=0.d0  ! cd is used to store also the HT data
!        outp%HFd1=0.d0
!        if(isot=='HD') then
!          !outp%HFqd=0.d0
!          outp%HFd2=0.d0
!        endif
!        do i=1,N
!          if(isot!='D2') outp%HFcp=outp%HFcp+chi(i)*p%HFcp(i)*chi(i)
!          if(isot!='H2') outp%HFcd=outp%HFcd+chi(i)*p%HFcd(i)*chi(i)
!          outp%HFd1=outp%HFd1+chi(i)*p%HFd1(i)*chi(i)
!          if(isot=='HD') then
!            !outp%HFqd=outp%HFqd+chi(i)*p%HFqd(i)*chi(i)
!            outp%HFd2=outp%HFd2+chi(i)*p%HFd2(i)*chi(i)
!          endif
!        enddo
!        print '(2x,a)','Hyperfine parameters   Value       Error'
!        print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cp > ",outp%HFcp,abs(outp%HFcp/pm)
!        if(isot=='HD') then
!          print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< cd > ",outp%HFcd,abs(outp%HFcd/pm)
!        elseif(isot=='HT') then
!          print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< ct > ",outp%HFcd,abs(outp%HFcd/pm)
!        endif
!        print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d1 > ",outp%HFd1,abs(outp%HFd1/pm)
!        if(isot=='HD') then
!          !print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< qD > ",outp%HFqd,abs(outp%HFqd/pm)
!          print '(6x,a7,10x,f10.5,2x,1pe9.1)',"< d2 > ",outp%HFd2,abs(outp%HFd2/pm)
!        endif
!        print '(1x,23a2)',('- ',i=1,23)
!      endif
!----------------------------------------
    IF(nonrel.eq."B") then !Nonrelativistic energy in BO approximation E(2,0), if you want it for some reason
       outp%E2=W(vstate)*u
       outp%E2napt=outp%E2
       outp%E2which="BO  "
    ELSE IF (nonrel.ne."F") then
       !DVR CALCULATION: NONADIABATIC, confer Eq. 53
       DO I=1,N
        DO J=1,I
         K = I*(I-1)/2+J
         IF(I.EQ.J) THEN
          H(K)= (one/(2*ma)+p%Wpara(I)/(pm**2))/dr**2*(pi**2/3-one/(2*I**2))&
              +one/(2*pm**2)*p%d2Wpara(I)                                    &
              +(one/(2*ma)+p%Worto(I)/(pm**2))*L*(L+1)/(p%R(I)**2)             &
              +p%V(I)+(p%Vad(I))/(2*pm)                                      &
              +p%VEna(I)/(pm**2)+p%dWpara(I)/(p%R(I)*pm**2)+p%VEnaprim(I)


         ELSE
           H(K) = (-1)**MOD(I-J,2)/(2*dr**2)*(2*one/(I-J)**2-2*one/(I+J)**2)*(one/ma+p%Wpara(J)/pm**2+p%Wpara(I)/pm**2)
         ENDIF
        ENDDO
       ENDDO

        IL=vstate
        IU=vstate
        CALL DSPEVX('N','I','U',N,H,VL,VU,IL,IU,ABSTOL,IE,Wn,Zn,N, WORK,IWORK,IFAIL,INFO)
      outp%E2=Wn(1)*u !Nonrelativistic energy E(2) via NAPT (see Section III B)
      outp%E2napt=outp%E2
      outp%E2which="NAPT"
   ENDIF

      if((nonrel.eq."A").or.(nonrel.eq."F")) then
       call check_nonad(qv,qj,isot,nonad_present,nonad_E2,u) !Looking for four-body values (see Section III A)
       if(nonad_present.eqv..true.) then
        outp%E2=nonad_E2
        outp%E2full=nonad_E2
        outp%E2which="FULL"
       endif
      endif

!Ena=E(2,2) nonadiabatic contribution (for error estimation)
outp%Ena=0.d0
do i=1,N
outp%Ena=outp%Ena+chi(i)*p%Vad(i)*chi(i) !This is the "adiabatic correction" Ead, stored only temporarily
enddo
outp%Ena=outp%E2napt-W(vstate)*u-outp%Ena*u/(2*pm) !E(2,2)=E(2)-E(2,0)-Ead

!Heteronuclear correction -- for HD, DT, and HT (expectation value of Eq. 37)
!For homonuclear molecules p%VEnaprim contains zeros
outp%Ehet=0.d0
do i=1,N
outp%Ehet=outp%Ehet+chi(i)*p%VEnaprim(I)*chi(i)
enddo
outp%Ehet=outp%Ehet*u
!----------------------------------------
!Relativistic corrections (BO&nonadiabatic)
      ma4=0.d0 
      do i=1,N
      ma4=ma4+chi(i)*p%ma4(i)*chi(i) !E(4,0) -- Eq. 61
      enddo

      ma4na=0.d0
      do i=1,N
      ma4na=ma4na+chi(i)*p%ma4na(i)*chi(i) !First-order part of E(4,1) -- Eq. 68, first line
      enddo

      ma4sec=0.d0
      do j=1,N
      if (j.eq.vstate) cycle
      tmp1=0.d0
      do i=1,N
      tmp1=tmp1+chi(i)*p%Vad(i)*Z(i,j)
      enddo
      tmp2=0.d0
      do i=1,N
      tmp2=tmp2+chi(i)*p%ma4(i)*Z(i,j)
      enddo
      ma4sec=ma4sec+1/(W(vstate)-W(j))*tmp1*tmp2 !Second-order part of E(4,1) -- Eq. 68, second line
      enddo

      outp%E4bo=(ma4)*u*alpha**2
      outp%E4na=(ma4na/(2*pm))*u*alpha**2
      outp%E4sec=(2*ma4sec/(2*pm))*u*alpha**2
      outp%E4=outp%E4bo+outp%E4na+outp%E4sec !Total E(4) contribution
!----------------------------------------
!Leading QED correction E(5) -- Eq. 69
       ma5=0.d0
       do i=1,N
       ma5=ma5+chi(i)*(p%ma5(i))*chi(i)
       enddo
       outp%E5=ma5*u*alpha**3
!----------------------------------------
!HQED E(6) correction E(6,0) -- Eq. 74
      ma6=0.d0
      do i=1,N
      ma6=ma6+chi(i)*p%ma6(i)*chi(i)
      enddo
      outp%E6bo=ma6*u*alpha**4 !First line of Eq. 74

      ma6sec=0.d0
      do j=1,N
      if (j.eq.vstate) cycle
      tmp1=0.d0
      do i=1,N
      tmp1=tmp1+chi(i)*p%ma4(i)*Z(i,j)
      enddo
      ma6sec=ma6sec+1/(W(vstate)-W(j))*tmp1**2
      enddo
      outp%E6sec=ma6sec*alpha**4*u !Second line of Eq. 74
!Total HQED E(6,0):
      outp%E6=outp%E6bo+outp%E6sec
!----------------------------------------
!Electron-nucleus Dirac deltas for E(7) and E(4)_FS corrections
      deltaia=0.d0
      do i=1,N
      deltaia=deltaia+chi(i)*p%deltaia(i)*chi(i)
      enddo
      ma7=-0.858549d0*alpha**5*Log(1/alpha**2)**2*deltaia*u !HQED E(7) -- Eq. 75
      outp%E7=ma7

      fs=2*pi/3*deltaia*ns*u !Finite-nucleus-size correction E(4)_FS -- Eq. 77
      outp%Efs=fs
!----------------------------------------
!Total energy and error evaluation
      outp%Etot=outp%E2+outp%E4+outp%E5+outp%E6+outp%E7+outp%Efs 
      outp%error_switch=0
      call errors1(outp)
!----------------------------------------

      END

!-----------------------------------------------------------------------------------------------

subroutine check_nonad(qv,qj,isot,nonad_present,nonad_E2,u)
!Checks if a fully nonadiabatic (four-body) energy is available for
!a given level -- in a file "<isot>.dat".
implicit none
double precision,intent(out) :: nonad_E2
double precision,intent(in) :: u
integer,intent(in) :: qv,qj
logical,intent(out) :: nonad_present
character(len=2),intent(in)::isot
character(len=11) :: filename
logical :: fileexists
integer:: v,j,stat
double precision :: En,SA,mprot,mdeut,mtrit
!It shouldn't matter for the accuracy of the results,
!but the fully nonadiabatic results for E(2) were calculated 
!with the old (CODATA 2014) nuclear masses, taken from:
!P. J. Mohr, D. B. Newell, and B. N. Taylor,
!Rev. Mod. Phys. 88, 035009 (2016)
!Thus the asymptotic values are calculated with those:
parameter (mprot = 1836.15267389D0) !Proton mass
parameter (mdeut = 3670.48296785D0) !Deuteron mass
parameter (mtrit = 5496.92153588D0) !Triton mass


      select case(isot)
      case("H2")
          SA=-mprot/(mprot+1) !2*reduced mass of an atom
      case("D2")
          SA=-mdeut/(mdeut+1)
      case("T2")
          SA=-mtrit/(mtrit+1)
      case("HD")
          SA=-(mprot/(mprot+1)+mdeut/(mdeut+1))/2
      case("HT")
          SA=-(mprot/(mprot+1)+mtrit/(mtrit+1))/2
      case("DT")
          SA=-(mdeut/(mdeut+1)+mtrit/(mtrit+1))/2
      end select

      nonad_present=.false.
      filename="data/"//isot//".dat"
      inquire(file=filename,exist=fileexists)
      if (fileexists.eqv..true.) then
      OPEN(UNIT=9,FILE=filename)
      do
      read(9,*,iostat=stat)v,j,En
      if (stat /= 0) exit
      if (v==qv.and.j==qj)then
       nonad_E2=SA-En
       nonad_E2=-nonad_E2*u
       nonad_present=.true.
       exit
      endif
      enddo
      close(9)
      endif 
end

!-----------------------------------------------------------------------------------------------
