module h2spectr_types
implicit none

!------------------------------------------------------------------------
!Derived types:
!A container type (potentials), containing all potential-type functions.
!Respective potentials described in './data/h2spectr_pot.f90' 
!in subroutine 'loader'.
type potentials
      double precision,allocatable :: V(:),Vad(:),ma4(:),ma5(:),ma6(:),ma4na(:)
      double precision,allocatable :: VEna(:),Wpara(:),dWpara(:),d2Wpara(:),Worto(:)
      double precision,allocatable :: deltaia(:),VEnaprim(:),R(:)
      double precision,allocatable :: HFcp(:),HFcd(:),HFd1(:),HFqd(:),HFd2(:)
end type

!A derived type containing all the calculated energy
!contributions, as well as their uncertainties.
!Explained in "h2spectr_level.f90".
type output
      double precision :: E2,E2full,E2napt,E4,E4na,E4bo,E4sec,E5,E6,E6bo,E6sec,E7,Efs,Etot,Ehet
      double precision :: errE2,errE4,errE5,errE6,errE7,errEfs,errEtot
      character(len=4) :: E2which
      double precision :: pm, Ena, rerr
      double precision,allocatable ::wav(:,:)
      integer :: error_switch
      double precision :: HFcp,HFcd,HFd1,HFqd,HFd2
end type

!------------------------------------------------------------------------

!Simple operations defined on the "output" type.
!The functions/subroutines are defined further.
interface assignment (=)
 module procedure eqo
end interface

interface operator (+)
 module procedure addo
end interface

interface operator (-)
 module procedure nego
 module procedure subo
end interface

!------------------------------------------------------------------------

contains

subroutine allocpot(p,NV)
!Allocate the "potentials" container
implicit none
integer,intent(in)::NV
type(potentials),intent(inout):: p
allocate(p%V(NV))
allocate(p%Vad(NV))
allocate(p%ma4(NV))
allocate(p%ma5(NV))
allocate(p%ma6(NV))
allocate(p%ma4na(NV))
allocate(p%VEna(NV))
allocate(p%Wpara(NV))
allocate(p%dWpara(NV))
allocate(p%d2Wpara(NV))
allocate(p%Worto(NV))
allocate(p%deltaia(NV))
allocate(p%VEnaprim(NV))
allocate(p%R(NV))
allocate(p%HFcp(NV))
allocate(p%HFcd(NV))
allocate(p%HFd1(NV))
allocate(p%HFqd(NV))
allocate(p%HFd2(NV))
end 

subroutine update(oin,contr,val,errval)
!Updates the "contr" contribution of the "oin" output variable
!with the "val" value and the "errval" uncertainty.
!The total energy and its uncertainty are updated automatically.
!Note that in the current version of the code, the "errval" given
!here can be overwritten by the automatic error estimation 
!procedure when the updated level is added/subtracted to/from another
!(see the "addo" and "errors1" subroutines below).
implicit none
type(output),intent(inout)::oin
double precision,intent(in) :: val,errval
character(len=*),intent(in)::contr
select case(trim(contr))
case ("E2")
oin%Etot=oin%Etot-oin%E2
oin%E2=val
oin%Etot=oin%Etot+oin%E2
oin%errE2=errval
oin%E2which="FULL" !Assumes that E2 is updated with a fully nonadiabatic result (why else do that?)
case ("E4")
oin%Etot=oin%Etot-oin%E4
oin%E4=val
oin%Etot=oin%Etot+oin%E4
oin%errE4=errval

case ("E5")
oin%Etot=oin%Etot-oin%E5
oin%E5=val
oin%Etot=oin%Etot+oin%E5
oin%errE5=errval

case ("E6")
oin%Etot=oin%Etot-oin%E6
oin%E6=val
oin%Etot=oin%Etot+oin%E6
oin%errE6=errval

case ("E7")
oin%Etot=oin%Etot-oin%E7
oin%E7=val
oin%Etot=oin%Etot+oin%E7
oin%errE7=errval

case ("FS")
oin%Etot=oin%Etot-oin%Efs
oin%Efs=val
oin%Etot=oin%Etot+oin%Efs
oin%errEfs=errval

case ("fs")
oin%Etot=oin%Etot-oin%Efs
oin%Efs=val
oin%Etot=oin%Etot+oin%Efs
oin%errEfs=errval

case default
write(*,*)"Unrecognised contribution name!"
end select
oin%errEtot=sqrt(oin%errE2**2+oin%errE4**2+oin%errE5**2+oin%errE6**2+oin%errE7**2+oin%errEfs**2)

end subroutine

!------------------------------------------------------------------------
subroutine errors1(oin)
!Calculates error estimates of "oin".
!For details, confer
!J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019)
!E(2) estimate based on calculation method.
USE,INTRINSIC :: IEEE_ARITHMETIC
implicit none
type(output),intent(inout)::oin
select case(oin%E2which)
case("NAPT")
oin%errE2=sqrt((oin%Ena/oin%pm)**2+(oin%Ehet*1.d-2)**2)
!It is particularly difficult to achieve good accuracy for the
!heteronuclear contribution (Eq. 37), so Ehet introduces 
!an additional numerical error.
!Due to the small magnitude of this correction itself, 
!it should be negligible, however.
case("BO")
oin%errE2=IEEE_VALUE(oin%errE2,IEEE_QUIET_NAN)
case("FULL")
oin%errE2=1.d-7
case default
oin%errE2=IEEE_VALUE(oin%errE2,IEEE_QUIET_NAN)
end select
!Errors estimated as the respective contribution times 1/pm - due to the neglected higher
!finite-nuclear-mass effects.
!E(4), E(5), and E(6) contain an additional numerical error to represent
!the uncertainty of the respective potential (which can be less reliable for higher levels).
!The formula for E(7) is known only roughly, hence the conservative 25% error estimate.
!E_FS contains a relative error from radius uncertainty.

oin%errE4=sqrt((oin%E4bo*2.d-6)**2+((oin%E4na+oin%E4sec)/oin%pm)**2+(oin%E4na*2.d-4)**2)
oin%errE5=sqrt((oin%E5/oin%pm)**2+(oin%E5*5.d-4)**2)
oin%errE6=sqrt((oin%E6/oin%pm)**2+(oin%E6bo*3.d-3)**2)
oin%errE7=abs(oin%E7)*0.25d0
oin%errEfs=sqrt((oin%Efs/oin%pm)**2+(oin%Efs*oin%rerr)**2)
oin%errEtot=sqrt(oin%errE2**2+oin%errE4**2+oin%errE5**2+oin%errE6**2+oin%errE7**2+oin%errEfs**2)
end subroutine

!------------------------------------------------------------------------
subroutine errors2(oin1,oin2,oout)
USE,INTRINSIC :: IEEE_ARITHMETIC
implicit none
type(output),intent(in)::oin1,oin2
type(output),intent(inout)::oout
select case(oout%error_switch)
case(1)
!Force the proportional error estimaton for transitions.
!Can lead to lesser error estimate, but it is less reliable
!when the radial wavefunctions differ much. It should be safe
!(and preferable) for transitions between closely-lying levels,
!such as (0,1)->(0,0).
call errors1(oout)

case default
!A more conservative error estimation method for transitions.
!For each energy contribution, the error is estimated as the bigger error
!from the pair of levels taking part in the transition.
if (oin1%errE2.gt.oin2%errE2) then
 oout%errE2=oin1%errE2
else
 oout%errE2=oin2%errE2
endif
if (oin1%errE4.gt.oin2%errE4) then
 oout%errE4=oin1%errE4
else
 oout%errE4=oin2%errE4
endif
if (oin1%errE5.gt.oin2%errE5) then
 oout%errE5=oin1%errE5
else
 oout%errE5=oin2%errE5
endif
if (oin1%errE6.gt.oin2%errE6) then
 oout%errE6=oin1%errE6
else
 oout%errE6=oin2%errE6
endif
if (oin1%errE7.gt.oin2%errE7) then
 oout%errE7=oin1%errE7
else
 oout%errE7=oin2%errE7
endif
if (oin1%errEfs.gt.oin2%errEfs) then
 oout%errEfs=oin1%errEfs
else
 oout%errEfs=oin2%errEfs
endif
oout%errEtot=sqrt(oout%errE2**2+oout%errE4**2+oout%errE5**2+oout%errE6**2+oout%errE7**2+oout%errEfs**2)


end select
end subroutine
!------------------------------------------------------------------------

subroutine eqo(oout,oin)
!Copies the "oin" output type to "oout".
implicit none
type(output),intent(in)::oin
type(output),intent(out)::oout
oout%E2      =oin%E2
oout%E2full  =oin%E2full
oout%E2napt  =oin%E2napt
oout%E4      =oin%E4
oout%E4bo    =oin%E4bo
oout%E4na    =oin%E4na
oout%E4sec   =oin%E4sec
oout%E5      =oin%E5
oout%E6      =oin%E6
oout%E6bo    =oin%E6bo
oout%E6sec   =oin%E6sec
oout%E7      =oin%E7
oout%Efs     =oin%Efs
oout%Etot    =oin%Etot
oout%E2which =oin%E2which
oout%pm      =oin%pm
oout%errE2   =oin%errE2
oout%errE4   =oin%errE4
oout%errE5   =oin%errE5
oout%errE6   =oin%errE6
oout%errE7   =oin%errE7
oout%errEfs  =oin%errEfs
oout%errEtot =oin%errEtot
oout%Ena     =oin%Ena
oout%rerr    =oin%rerr
allocate(oout%wav(size(oin%wav,1),2))
oout%wav     =oin%wav
oout%Ehet=oin%Ehet
oout%error_switch=oin%error_switch
end subroutine

!------------------------------------------------------------------------

function nego(oin) result(oout)
!Inverses the sign of all the "oin" components, saving the result in "oout".
implicit none
type(output),intent(in)::oin
type(output)::oout
oout%E2     =-oin%E2
oout%E2full =-oin%E2full
oout%E2napt =-oin%E2napt
oout%E4     =-oin%E4
oout%E4bo   =-oin%E4bo
oout%E4na   =-oin%E4na
oout%E4sec  =-oin%E4sec
oout%E5     =-oin%E5
oout%E6     =-oin%E6
oout%E6bo   =-oin%E6bo
oout%E6sec  =-oin%E6sec
oout%E7     =-oin%E7
oout%Efs    =-oin%Efs
oout%Etot   =-oin%Etot
oout%E2which= oin%E2which
oout%pm     = oin%pm
oout%errE2  = oin%errE2
oout%errE4  = oin%errE4
oout%errE5  = oin%errE5
oout%errE6  = oin%errE6
oout%errE7  = oin%errE7
oout%errEfs = oin%errEfs
oout%errEtot= oin%errEtot
oout%Ena    =-oin%Ena
oout%rerr   = oin%rerr
allocate(oout%wav(size(oin%wav,1),2))
oout%wav    =oin%wav
oout%Ehet=-oin%Ehet
oout%error_switch=oin%error_switch
end function

!------------------------------------------------------------------------

function subo(oin1,oin2) result(oout)
!Subtracts "oin2" from "oin1", saving the result in "oout".
USE,INTRINSIC :: IEEE_ARITHMETIC
implicit none
type(output),intent(in)::oin1,oin2
type(output)::oout
type(output)::otmp
otmp=-oin2
oout=oin1+otmp
end function

!------------------------------------------------------------------------

function addo(oin1,oin2) result(oout)
!Adds "oin2" to "oin1", saving the result in "oout",
!checking consistency of "pm" and "E2which".
USE,INTRINSIC :: IEEE_ARITHMETIC
implicit none
type(output),intent(in)::oin1,oin2
type(output)::oout
if(oin1%pm.ne.oin2%pm) then
write(*,*)"ERROR! Reduced nuclear masses are different!"
write(*,*)"Aren't you trying to combine levels of different isotopologues?"
stop
endif
!If four-body results are available for both levels, use them.
!If not, use NAPT results. If (for some reason) one of the levels is just 
!in BO approximation, return NaN. If both are BO, use them.
if(oin1%E2which=="FULL".and.oin2%E2which=="FULL")then
oout%E2   =oin1%E2full+oin2%E2full
oout%E2which="FULL"
else if(oin1%E2which=="NAN ".or.oin2%E2which=="NAN ") then
oout%E2   =IEEE_VALUE(oout%E2,IEEE_QUIET_NAN)
oout%E2which="NAN "
else if((oin1%E2which=="BO  ".and.oin2%E2which.ne."BO  ")&
       .or.(oin1%E2which.ne."BO  ".and.oin2%E2which=="BO  "))then
oout%E2   =IEEE_VALUE(oout%E2,IEEE_QUIET_NAN)
oout%E2which="NAN "
else if(oin1%E2which=="BO  ".and.oin2%E2which=="BO  ")then
oout%E2   =oin1%E2napt+oin2%E2napt
oout%E2which="BO  "
else
oout%E2   =oin1%E2napt+oin2%E2napt
oout%E2which="NAPT"
endif
oout%E2napt=oin1%E2napt+oin2%E2napt
oout%E2full=oin1%E2full+oin2%E2full
oout%E4    =oin1%E4    +oin2%E4
oout%E4bo  =oin1%E4bo  +oin2%E4bo
oout%E4na  =oin1%E4na  +oin2%E4na
oout%E4sec =oin1%E4sec +oin2%E4sec
oout%E5    =oin1%E5    +oin2%E5
oout%E6    =oin1%E6    +oin2%E6
oout%E6bo  =oin1%E6bo  +oin2%E6bo
oout%E6sec =oin1%E6sec +oin2%E6sec
oout%E7    =oin1%E7    +oin2%E7
oout%Efs   =oin1%Efs   +oin2%Efs
oout%pm    =oin1%pm
oout%rerr  =oin1%rerr
oout%Ena   =oin1%Ena+oin2%Ena
oout%Etot=oout%E2+oout%E4+oout%E5+oout%E6+oout%E7+oout%Efs
allocate(oout%wav(size(oin1%wav,1),2))
oout%wav=IEEE_VALUE(oout%wav,IEEE_QUIET_NAN)
oout%Ehet=oin1%Ehet+oin2%Ehet
oout%error_switch=oin1%error_switch
call errors2(oin1,oin2,oout)
end function



!------------------------------------------------------------------------
end module

module control_parameters
implicit none

logical*1 HF_bool
character*2 :: molecule

end module
