PROGRAM H2SPECTRE
!The main 'black-box' executable, taking calculation parameters from the user and issuing
! the 'level' subroutine.
!Equation numbers refer to J. Komasa, M. Puchalski, P. Czachorowski, G. Lach, K. Pachucki, Phys. Rev. A 100, 032519 (2019),
!unless stated otherwise. 
      use h2spectr_types
      use control_parameters
      implicit none
      type(output) :: d0,level1,level2
      type(output),allocatable :: level_database(:)
      integer,allocatable:: db_index(:,:)
      integer :: indx,indx2,db_size,stat,error_switch
      logical :: db_flag
      character*2 :: tl
      character*30 :: keyword
      character*120::str
      double precision :: ran,na_cutoff
      character(len=1)::nonrel
      integer :: v1,J1,v2,J2,io,cnt,prnlev,N,i,n_cmd_args,prec,units
      logical :: trans,valid,saving,fileexists,prec_bool,valid_overr,wavl
      character(len=30),dimension(:),allocatable :: cmd_args

!Default parameters
  nonrel="A" !Check for fully nonadiabatic results, use NAPT if not found.
  N=200      !DVR grid points
  ran=10.d0  !Integration range (R_m for m=N, from Eq. 44)
  prnlev=0 !Output verbosity level (0/1/2)
  units=0 !Output units (0: cm-1, 1: MHz)
  wavl=.false. !Print wavelength or not
  saving=.true. !Save calculated level energies to reuse for multiple transitions?
  prec_bool=.false. !Output precision override
  error_switch=0 !Error estimation method for transitions
  valid_overr=.false. !Whether to override the level validity check
  na_cutoff=5.d-2 !Nonadiabatic heteronuclear potential cutoff distance
  HF_bool=.false.

!Read command-line arguments...
  n_cmd_args=command_argument_count() !Check how many of them are present
  allocate(cmd_args(n_cmd_args))
  do i=1,n_cmd_args
   call get_command_argument(i,cmd_args(i))
  enddo

!...and parse them
  do i=1,n_cmd_args

   select case (trim(cmd_args(i)))
   case('--pc','--PC')
   !Just query the 'level' subroutine for the constants it uses, then exit.
    write(*,'(a/)') "Physical constants used:"
    call level(0,0,'PC','B',0,0.d0,0.d0,level1)
   
   case('-N') !Change the number of grid points.
    read(cmd_args(i+1),*,err=999)N
  
   case('-R') !Change the integration range.
    read(cmd_args(i+1),*,err=998)ran
   
   case('-V') !Change verbosity of the output.
    read(cmd_args(i+1),*,err=997)prnlev

   case('-P') !Change the number of the output digits displayed.
    read(cmd_args(i+1),*,err=996)prec
    prec_bool=.true.

   case('-E') !Change the error estimation method.
    read(cmd_args(i+1),*,err=995)error_switch

   case('--cutoff') !Change the cutoff for the heteronuclear nonadiabatic potential.
    read(cmd_args(i+1),*,err=994)na_cutoff

   case('--napt', '--NAPT') !Force NAPT (ignores fully nonadiabatic values).
    nonrel="N"

   case('--wavl') !Print the wavelength in micrometers.
    wavl=.true.

   case('--MHz', '--mhz') !Print the output in MHz, instead of default cm-1
    units=1

   case('-S','-s') !Enable level saving (default, kept for backwards compatibility).
    saving=.true.

   case('-Sn','-sn') !Disable level saving.
    saving=.false.
 
   case('-Dv','-dv') !Override level validity check
    saving=.false.
    valid_overr=.true.

   case('--HF', '--hf') ! Perform averaging of hyperfine parameters
    HF_bool=.true.

   case('-h','--help') !Get help -- print the first part of 'README.txt', which
                       !contains an explanation of the command-line options
      inquire(file='README.txt',exist=fileexists)
      if (fileexists.eqv..true.) then
      OPEN(UNIT=9,FILE='README.txt')
      do
       read(9,'(A)',iostat=stat)str
       if ((stat /= 0).or.(trim(str).eq."Contents of this directory:")) exit
       write(*,*)str
      enddo
      close(9)
      else
      write(*,*)"README.txt not found for some reason..."
      endif
      stop

   case default
    !Numerical parameters are OK only if they are after to -N, -R, -V, -P, -E or --cutoff.
    if(trim(cmd_args(i-1)).eq.'-N'.or.trim(cmd_args(i-1)).eq.'-R'.or.trim(cmd_args(i-1)).eq.'-V'.or.trim(cmd_args(i-1)).eq.'-P'.or.trim(cmd_args(i-1)).eq.'-E'.or.trim(cmd_args(i-1)).eq.'--cutoff') then
     cycle
    else
    write(*,*)"Wrong command-line parameter! Exiting."
    write(*,*)"  ",trim(cmd_args(i))
    stop
    endif
   end select
  enddo
  goto 1000

!IO exception handling
999 write(*,*)"Wrong -N parameter! Exiting."
    write(*,*)cmd_args(i+1)
    stop
998 write(*,*)"Wrong -R parameter! Exiting."
    write(*,*)cmd_args(i+1)
    stop
997 write(*,*)"Wrong -V parameter! Exiting."
    write(*,*)cmd_args(i+1)
    stop
996 write(*,*)"Wrong -P parameter! Exiting."
    write(*,*)cmd_args(i+1)
    stop
995 write(*,*)"Wrong -E parameter! Exiting."
    write(*,*)cmd_args(i+1)
    stop
994 write(*,*)"Wrong --cutoff parameter! Exiting."
    write(*,*)cmd_args(i+1)
    stop


1000 continue
! Read control data from stdin
  read(*,*) molecule,keyword

! Echo input control line and print info about number of points and integration range
  if(molecule=='H2'.or.molecule=='HD'.or.molecule=='HT'.or.molecule=='D2'.or.molecule=='DT'.or.molecule=='T2') then
  if(prnlev.gt.0) then
    write(*,'(a13,i5,1x,a13,f5.1,a1)')"(Int. points=",N,", Int. range=",ran,")"
    write(*,*)"---------------------------------------------"
  endif
    write(*,'(1x,a2,5x,a30)') molecule,keyword
  else
    stop "Incorrect isotopologue"
  endif

  tl=keyword(1:2)
  if(tl=='TR' .or. tl=='tr' .or. tl=='Tr') then
    trans=.true.
  elseif(tl=='LE' .or. tl=='le' .or. tl=='Le') then
    trans=.false.
  else
    stop ': Incorect keyword! Use TR[ANSITION] or LE[VEL]'
  endif

! Hyperfine parameters implemented for H2, D2, HD and HT only
  if(HF_bool .and. ((molecule/='HD' .and. molecule/='H2' .and. molecule/='D2' .and. molecule/='HT') &
     .or. ((molecule=='HD' .or. molecule=='H2' .or. molecule=='D2' .or. molecule=='HT') .and. trans))) then
    print *,'Hyperfine parameters implemented only for the Levels of H2, D2, HD and HT'
    stop
  endif

!Allocate the database to store calculated levels for further use
    if(trans.and.saving) then
    select case(molecule)
    case("H2")
     db_size=302
    case("D2")
     db_size=601
    case("T2")
     db_size=897
    case("HD")
     db_size=400
    case("HT")
     db_size=449
    case("DT")
     db_size=720
    case default
     stop
    end select
    allocate(level_database(db_size),db_index(db_size,2))
    indx2=0
    endif

!Loop over levels or transitions
  do
    db_flag=.false. !Flag denoting if the level has been previously calculated (and can be reused).
    J1=-1
    if(trans) then
      read(*,*,iostat=io) v1,J1,v2,J2
    else
      read(*,*,iostat=io) v1,J1
    endif
    if(io.ne.0 .or. J1.lt.0) exit

    write(*,'(/1x,a5,i2,3x,a5,i2)',advance='no') "v' = ",v1,"J' = ",J1
    if(trans) then
      write(*,'(1x,a9,a5,i2,3x,a5,i2)') "   -->   ","v"" = ",v2,"J"" = ",J2
    else
      write(*,*)
    endif

!Check validity of the quantum numbers entered
    valid=.true.
    call check_vJ(v1,J1,valid)
    if(trans) call check_vJ(v2,J2,valid)
    if(.not.valid) then
      write(*,*) "A quantum number seems too big..."
      if (.not.valid_overr) exit !By default it is not only a warning, but an error
    endif


!Evaluate the levels
    if (trans.and.saving) call check_db(v1,J1,db_flag,indx,indx2) !Check if the level is already available
    if (db_flag) then
     level1=level_database(indx)
    else
    call level(v1,J1,molecule,nonrel,N,ran,na_cutoff,level1)
    level1%error_switch=error_switch
    if(trans.and.saving) then
    indx2=indx2+1
    db_index(indx2,1)=v1
    db_index(indx2,2)=J1
    level_database(indx2)=level1
    endif
    endif
 
    if(.not.trans) then
 
      d0=-level1
      call prnt
 
    else


    if (trans.and.saving) call check_db(v2,J2,db_flag,indx,indx2)
    if (db_flag) then
     level2=level_database(indx)
    else
      call level(v2,J2,molecule,nonrel,N,ran,na_cutoff,level2)
      level2%error_switch=error_switch
    if(trans.and.saving) then
    indx2=indx2+1
    db_index(indx2,1)=v2
    db_index(indx2,2)=J2
    level_database(indx2)=level2
    endif
    endif

      d0=subo(level1,level2)
      call prnt
 
    endif

  enddo

contains
  subroutine check_db(v,J,db_flag,indx,indx2)
  !Query the database of already calculated levels
  implicit none
  integer,intent(in)::v,J
  logical,intent(out)::db_flag
  integer,intent(out)::indx
  integer,intent(in)::indx2
  db_flag=.false.
  do i=1,indx2
  if((db_index(i,1).eq.v).and.(db_index(i,2).eq.J)) then
   indx=i
   db_flag=.true.
   exit
  endif
  enddo
  end


  subroutine prnt
  !The printing subroutine
    character :: form0*34,form1*34,form2*24,form3*28,wdth*2,wdth2*2,unitchar*4
    integer :: ndd,minndd !ndd - number of digits after decimal dot
    double precision :: upref !units prefactor

    select case(units) 
    case(1)
     upref=29979.2458d0 !Speed of light times 10^-4, cm-1 -> MHz
     unitchar="MHz "
     minndd=1
    case default
     upref=1.d0
     unitchar="cm-1"
     minndd=5
    end select
  
    !Output precision
    ndd=ceiling(-log10(upref*d0%errEtot))+1
    ndd=max(minndd,min(89,ndd))
    if(prec_bool) then
     ndd=min(89,prec)
    endif

    write(wdth,'(I2)')max(1,ndd)

    form0='(A14,6x,A6,A4,A1,'
    form0=trim(form0)//trim(adjustl(wdth))//'x,A6,A4,A1)'
    
    write(wdth,'(I2)')ndd
    write(wdth2,'(I2)')ndd+13

    form1='(6x,a3,a4,a1,1x,f'
    form1=trim(form1)//trim(adjustl(wdth2))//'.'//trim(adjustl(wdth))//',1x,1pe10.1)'

    form2='(a15,f'
    form2=trim(form2)//trim(adjustl(wdth2))//'.'//trim(adjustl(wdth))//',1x,1pe10.1)'

    form3='(a8,a4,a1,f'
    form3=trim(form2)//trim(adjustl(wdth2))//'.'//trim(adjustl(wdth))//',1x,1pe10.1)'

    if(prnlev>0) then
      write(*,form0)"Contribution","Value[",unitchar,"]","Error[",unitchar,"]"
      write(*,form1)"E2(",d0%E2which,")",upref*d0%E2,upref*d0%errE2
      write(*,form2)"E4       ",upref*d0%E4,upref*d0%errE4
      if(prnlev>1) then
        write(*,form2)"    E4(BO) ",upref*d0%E4bo
        write(*,form2)"    E4(rec)",upref*d0%E4na
        write(*,form2)"    E4(2nd)",upref*d0%E4sec
      endif
      write(*,form2)"E5       ",upref*d0%E5,upref*d0%errE5
      write(*,form2)"E6       ",upref*d0%E6,upref*d0%errE6
      if(prnlev>1) then
        write(*,form2)"    E6(BO) ",upref*d0%E6bo
        write(*,form2)"    E6(2nd)",upref*d0%E6sec
      endif
      write(*,form2)"E7       ",upref*d0%E7,upref*d0%errE7
      write(*,form2)"Efs      ",upref*d0%Efs,upref*d0%errEfs
    endif
    if ((.not.wavl).or.prnlev>0) then
    write(*,form2)"Total    ",upref*d0%Etot,upref*d0%errEtot
    else
    write(*,form3)"Energy[",unitchar,"]",upref*d0%Etot,upref*d0%errEtot
    endif
    write(*,*)"---------------------------------------------"   


    if (wavl) then !Wavelength in micrometers

    !Output precision
    ndd=ceiling(-log10(1.d4*d0%errEtot/(d0%Etot)**2))+1
    ndd=max(5,min(89,ndd))
    if(prec_bool) then
     ndd=min(89,prec)
    endif


    write(wdth,'(I2)')ndd
    write(wdth2,'(I2)')ndd+10
 
    form2='(a15,f'
    form2=trim(form2)//trim(adjustl(wdth2))//'.'//trim(adjustl(wdth))//',1x,1pe10.1)'
    write(*,form2)" Wavelength[um]",1.d4/d0%Etot,1.d4*d0%errEtot/(d0%Etot)**2
    write(*,*)"---------------------------------------------"




    endif
  end subroutine prnt

  subroutine check_vJ(v,J,valid)
  !Check in database if the rovibrational level (v,J) is bound and is even worth calculating.
    implicit none
    integer :: v,J,vmax,Jmax
    integer,parameter :: vmaxH2=14,vmaxHD=17,vmaxHT=18,vmaxD2=21,vmaxDT=23,vmaxT2=26
    integer,dimension(0:vmaxH2),parameter :: H2vJ=(/31,30,28,27,25,23,22,20,18,16,14,12,10,7,4/)
    integer,dimension(0:vmaxHD),parameter :: HDvJ=(/36,35,33,32,30,28,27,25,23,21,20,18,16,13,11,8,5,1/)
    integer,dimension(0:vmaxHT),parameter :: HTvJ=(/39,37,35,34,32,31,29,27,26,24,22,20,18,16,14,11,9,5,1/)
    integer,dimension(0:vmaxD2),parameter :: D2vJ=(/45,43,42,40,39,37,35,34,32,30,29,27,25,23,21,19,17,14,12,9,5,1/)
    integer,dimension(0:vmaxDT),parameter :: DTvJ=(/49,48,46,45,43,41,40,38,37,35,33,31,30,28,26,24,22,20,17,15,12,9,6,1/)
    integer,dimension(0:vmaxT2),parameter :: T2vJ=(/55,54,52,51,49,47,46,44,43,41,39,38,36,34,32,30,28,26,24,22,20,18,15,12,9,5,0/)
    logical :: valid

    select case(molecule)
      case('H2')
        if(v>vmaxH2) then; valid=.false.; return; endif
        Jmax=H2vJ(v)
      case('HD')
        if(v>vmaxHD) then; valid=.false.; return; endif
        Jmax=HDvJ(v)
      case('HT')
        if(v>vmaxHT) then; valid=.false.; return; endif
        Jmax=HTvJ(v)
      case('D2')
        if(v>vmaxD2) then; valid=.false.; return; endif
        Jmax=D2vJ(v)
      case('DT')
        if(v>vmaxDT) then; valid=.false.; return; endif
        Jmax=DTvJ(v)
      case('T2')
        if(v>vmaxT2) then; valid=.false.; return; endif
        Jmax=T2vJ(v)
      case('PC')
        J=0
      case default 
        write(*,*) "Invalid isotopologue!"
        stop
    end select

    if(J>Jmax) valid=.false.

    !Additional warning about the highest level for each isotopologue
    if (v==14.and.J==4.and.molecule=="H2".or.v==21.and.J==1.and.molecule=="D2".or.v==26.and.J==0.and.molecule=="T2".or.&
        v==17.and.J==1.and.molecule=="HD".or.v==23.and.J==1.and.molecule=="DT".or.v==18.and.J==1.and.molecule=="HT") then
        write(*,*)"Weakly bound level involved. Caution advised."
        write(*,*)"Error can be bigger than the estimate given."
    endif
    return
  end subroutine check_vJ

END


