ClearAll["Global`*"]
Quiet[SetDirectory[NotebookDirectory[]]];
(*
PbarSpectr v 1.0

Main program for EXTENDED size orbiting particle

Instructions for running Mathematica code:

The code was prepared in Mathematica 14.2.
Main file which should be run is PbarSpectr.m
In this file all the parameters (quantum numbers, masses, g-factor, charge radius, grid parameters etc.) are set.

All the other auxiliary files will be loaded automatically and should be inside the same directory as the main program.
After setting all parameters below the program is launched by command

<<PbarSpectr.m

from Mathematica in command line, or directly from graphical interface.

auxiliary files:

nrwfandenergy.m - file with main algorithm for numerical determination of nonrelativistic energy and wave function
nrwfandenergyFast.m - the same but with faster numerical evaluation of integration for Kallen-Sabry potential. Which file is
used is determined in PbarSpectr.m
E4s0.m - expectation value of Breit Hamiltonian which includes vacuum polarization
Ehigher.m - contribution to energies of the order alpha^5 and higher, hydrogenic results without vacuum polarization
gausext.m - fast numerical integration procedure used in nrwfandenergyFast.m
bethelog.m - table of values for Bethe logarithm for various excited states
WKpotential.m - code for Wichman-Kroll potential, used for solving Schrodinger equation
*)

(** USER DEFINED PARAMETERS **)

(* quantum numbers of the state under consideration *)

n = 6;
l = 5;
j = 11/2;


(* nuclear charge *)

z = 10;


(* mass ratios of electron to particles #1 (orbitting) and #2 (nucleus) *)

M1 = 1/1836.152673426;
M2 = 1/36433.99594496038777;

(* orbitting particle #1 g-factor  *)

g1 = SetPrecision[5.5856946893,20];	(* proton g factor *)

rC = 3.0055`20; 		(* 20^Ne nuclear charge radius in fm, can be kept arbitrary *)
rP = 0.84075`20;		(* proton charge radius in fm *)


(* grid parameters, as defined in Appendix *)

rO = 3*10^-4;
h = 3*10^-3;
numberOfSteps = 75000;

(* desired limit of the accuracy for the energy (discontinuity of Q in the turning point) *)

del = 10^-21;

(* limit on the ratio of Uehling potential/WK potential/two-loop vacuum polarization to Coulomb potential *)
(* if the ratio of vacuum polarization to Coulomb becomes smaller for particular r, then for larger radial distances r the vacuum polarization potentials are omitted *)

Vdel = 10^-21;

(* parameter KSINT to decide if slower but more accurate algorithm for two-loop vacuum polarization potential is used (set KSINT=1), or faster but less accurate algorithm is used (set KSINT=2) *)
(* in first case the file ntwfandenergy.m will be loaded, in second case the file nrwfandenergyFast.m will be loaded *)

KSINT = 2;

(* working precision in numerical integration of two-loop vacuum polarization potential, not used in nrwfandenergyFast.m algorithm *)

WP = 14;

(* number of steps in Adams-Moulton method *)

k = 8;

(** END OF USER DEFINED PARAMETERS **)



prec = 80;	(* precision used in calculation, for evaluation of analytical form of Uehling potential machine precision is not enough *)

(* spin-orbit interaction L.s1 of orbiting particle *)

LS1 = 1/2 (j(j+1) - l(l+1) - 3/4);

lC = 386.15926796`20; 		(* reduced Compton wavelength in fm *)

(* reduced atomic units are used in the code, where reduced mass is set mu=1 and electron mass is me=me/mu*)

(* electron mass me/mu = me/m1+me/m2 *)

m = SetPrecision[M1+M2,prec];

(* orbiting particle-nucleus mass ratio *)

mrat = SetPrecision[M2/M1,prec];

(* masses of both particles, in r.a.u. *)

m1 = 1+mrat;
m2 = m1/(m1-1);


(* fine structure constant, NIST 24 *)

alp = SetPrecision[0.0072973525643,prec];

(* a.u. to ev conversion factor, from NIST 25 *)

evfac = 27.211386245981`30;

(* a.u. to Hz conversion factor, from NIST 25 *)

hzfac = 6.5796839204999`30 10^15;

Print[""]
Print["PbarSpectr v1.0"]
Print[""]

(* solving Schrodinger equation with vacuum polarization and obtaining nonrelativistic energy and wf *)
Print["Generating values of the potential on a grid, it can take a few minutes"]
Print[""]
If[KSINT==1,<<nrwfandenergy.m,If[KSINT==2,<<nrwfandenergyFast.m]];


(* Breit Hamiltonian with exact vacuum polarization *)
<<E4s0.m;

(* higher order contributions *)
<<Ehigher.m;

(* fns contribution *)
EFS = SetPrecision[1/6 ( (rC/(M1 m1 lC))^2 + (rP/(M1 m1 lC))^2 ) ss1,20];
EFS = EFS/m alp^2 evfac;
Print["Finite nuclear size contribution, in eV"]
Print[EFS]
Print[""]

Print["____________________________"]

(* total energy *)

Print["Total energy in eV"];
Etot = energy/m*evfac + E4tot + E5tot + E6tot + E7tot + E8tot + EFS;
Print[NumberForm[Etot, 20]];
Print[""]

Print["Total uncertainty in eV"];

Eunc = Sqrt[E7unc^2+E8unc^2+E3l^2];

Print[NumberForm[Eunc, 20]];
Print[""]

Print["Total energy in GHz"];
Print[NumberForm[Etot hzfac/evfac 1/10^9, 20]];
Print[""]

Print["Total uncertainty in GHz"];
Print[NumberForm[Eunc hzfac/evfac 1/10^9, 20]];
