/**************************************************************************
*           Copyright (c) 2001, Cisco Systems, All Rights Reserved
***************************************************************************
*
*  File:    interceptor.c
*  Date:    04/10/2001
*
***************************************************************************
* This module implements the linux driver.
***************************************************************************/
#include <linux/version.h>
#include <linux/config.h>
#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/in.h>
#include <linux/ppp_defs.h>
#include <net/ip.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <net/protocol.h>
#include <net/dst.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/vmalloc.h>
#endif

#include "linux_os.h"
#include "vpn_ioctl_linux.h"
#include "Cniapi.h"
#include "linuxcniapi.h"

#include "frag.h"

/*methods of the cipsec network device*/
static int interceptor_init(struct device *);
static struct enet_statistics *interceptor_stats(struct device *dev);
static int interceptor_ioctl(struct device *dev, struct ifreq *ifr, int cmd);
static int interceptor_open(struct device *dev);
static int interceptor_stop(struct device *dev);
static int interceptor_tx(struct sk_buff *skb, struct device *dev);

/*helper functions*/
static BINDING *getbindingbydev(struct device *dev);
static int handle_vpnup();
static int handle_vpndown();
static CNIFRAGMENT build_ppp_fake_mac_frag(struct ethhdr *dummy);
static int inline supported_device(struct device* dev);
int inline ippp_dev(struct device* dev);

/*packet handler functions*/
static int recv_ip_packet_handler(struct sk_buff *skb,
                                  struct device *dev,
                                  struct packet_type *type);
static int replacement_dev_xmit(struct sk_buff *skb,
                                            struct device *dev);

#define PPP_HLEN 4

struct packet_type_funcs {
    struct packet_type *pt;
    int (*orig_handler_func) (struct sk_buff *, struct device *,
                                struct packet_type *);
};
static struct packet_type_funcs original_ip_handler;

extern CNI_CHARACTERISTICS CNICallbackTable; /* This stores the Plugin's function pointers */
extern PCHAR pcDeviceName;      /* Ignore. Only our pluggin so we don't care about it */

static int tx_packetcount = 0;
static int rx_packetcount = 0;
static int num_target_devices;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#define interceptor_names LINUX_VPN_IFNAME
#else
static char interceptor_names[8];
#endif

BINDING Bindings[MAX_INTERFACES];
BINDING BindingIpcUdp;

static struct device interceptor_devs[] = {
    {
     interceptor_names,
     0, 0, 0, 0,
     0x0000,
     0,
     0, 0, 0,
     NULL,
     interceptor_init}
};

int init_module(void)
{
    int status = 0;
    PCNI_CHARACTERISTICS PCNICallbackTable;
    CNISTATUS rc = CNI_SUCCESS;
    rc = CniPluginLoad(&pcDeviceName, &PCNICallbackTable);

    if (CNI_IS_SUCCESS(rc)) {

        CNICallbackTable = *PCNICallbackTable;
        CniPluginDeviceCreated();

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
        memcpy(interceptor_devs[0].name, LINUX_VPN_IFNAME,
               strlen(LINUX_VPN_IFNAME) + 1);

#endif
        if ((status = register_netdev(&interceptor_devs[0])) != 0) {
            printk(KERN_INFO "%s: error %d registering device \"%s\".\n",
                   LINUX_VPN_IFNAME, status, interceptor_devs[0].name);
            CniPluginUnload();

        }
    }
    if (status == 0) {
        printk(KERN_INFO "Cisco Systems VPN Client Version "
               BUILDVER_STRING " kernel module loaded\n");
    }
    return (status);
}

void cleanup_module(void)
{
    cleanup_frag_queue();
    CniPluginUnload();

    unregister_netdev(&interceptor_devs[0]);
    return;
}

static int interceptor_init(struct device *dev)
{
    dev->open = interceptor_open;
    dev->stop = interceptor_stop;
    dev->hard_start_xmit = interceptor_tx;
    dev->get_stats = interceptor_stats;
    dev->do_ioctl = interceptor_ioctl;


    dev->type = ARPHRD_ETHER;
    dev->hard_header_len = ETH_HLEN;
    dev->mtu = 1400;            /* eth_mtu */
    dev->addr_len = ETH_ALEN;
    dev->tx_queue_len = 100;    /* Ethernet wants good queues */

    dev->flags = IFF_BROADCAST | IFF_MULTICAST;
    memset(dev->broadcast, 0xFF, ETH_ALEN);

    return 0;
}

static struct enet_statistics *interceptor_stats(struct device *dev)
{
    static struct enet_statistics es;


    es.rx_packets = rx_packetcount;
    es.rx_errors = 0;
    es.rx_dropped = 0;
    es.rx_fifo_errors = 0;
    es.rx_length_errors = 0;
    es.rx_over_errors = 0;
    es.rx_crc_errors = 0;
    es.rx_frame_errors = 0;
    es.tx_packets = tx_packetcount;
    es.tx_errors = 0;
    es.tx_dropped = 0;
    es.tx_fifo_errors = 0;
    es.collisions = 0;
    es.tx_carrier_errors = 0;
    es.tx_aborted_errors = 0;
    es.tx_window_errors = 0;
    es.tx_heartbeat_errors = 0;

    return (&es);
}

static int interceptor_ioctl(struct device *dev, struct ifreq *ifr,
                             int cmd)
{
    int error = 0;
    static struct ifvpncmd command;


    switch (cmd) {
    case SIOCGVPNCMD:
        {
            uint32 ret_size;

            char *from = (char *) ifr->ifr_ifru.ifru_data;
            COPY_FROM_USER(error, &command, from, sizeof(struct ifvpncmd));

            error = CniPluginIOCTL(command.cmd, (void *) command.data,
                                   command.datalen, sizeof(command.data),
                                   &ret_size);

            if (!error) {
                command.datalen = ret_size;
            } else {
                command.datalen = 0;
            }

            COPY_TO_USER(error, from, &command, sizeof(struct ifvpncmd));
        }
        break;

    case SIOCGVPNIFUP:
        error = handle_vpnup();
        break;

    case SIOCGVPNIFDOWN:
        error = handle_vpndown();
        break;

    default:
        error = -EOPNOTSUPP;
        break;
    }

    return error;
}

static int interceptor_open(struct device *dev)
{
    /*stub. our network interface shouldn' ever come up*/
    return 1;
}

static int interceptor_stop(struct device *dev)
{
    /*stub. our network interface shouldn' ever come up*/
    return 1;
}

static int interceptor_tx(struct sk_buff *skb, struct device *dev)
{
    dev_kfree_skb(skb);
    return 0;
}

static int handle_vpnup()
{
    /*temporary structure used to retrieve the registered ip packet handler.
     *it is static because it gets inserted temporarily into a kernel hash
     *table and if things went incredibly wrong it could end up staying there
     */
    static struct packet_type dummy_pt;

    struct device *dp=NULL;
    struct packet_type *default_pt=NULL;
    int error = VPNIFUP_SUCCESS;

    cleanup_frag_queue();

    if (MOD_IN_USE) {
        error = VPNIFUP_FAILURE;
        goto exit_gracefully;
    }

    /* find the handler for inbound IP packets by adding a dummy handler
     * for that packet type into the kernel. Because the packet handlers
     * are stored in a hash table, we'll be able to pull the original 
     * ip packet handler out of the list that dummy_pt was inserted into.*/
    memset(&dummy_pt,0,sizeof(dummy_pt));
    dummy_pt.type =  htons(ETH_P_IP);
    dummy_pt.func = recv_ip_packet_handler;

    dev_add_pack(&dummy_pt);
    /* this should be the original IP packet handler*/
    default_pt = dummy_pt.next;
    /* there may be more than one other packet handler in our bucket,
     * so look through all the buckets */
    while (default_pt != NULL && default_pt->type != htons(ETH_P_IP))
    {
        default_pt = default_pt->next;
    }
    if (!default_pt)
    {
        printk(KERN_DEBUG "No default handler found for %x protocol!!\n",
               dummy_pt.type);
        dev_remove_pack(&dummy_pt);
        error = VPNIFUP_FAILURE;
        goto exit_gracefully;
    }
    /*remove the dummy handler handler*/
    original_ip_handler.pt = default_pt;
    dev_remove_pack(&dummy_pt);

    /*and replace the original handler function with our function*/
    original_ip_handler.orig_handler_func
            = original_ip_handler.pt->func;
    original_ip_handler.pt->func = recv_ip_packet_handler;

    /* identify the active network devices */
    memset(&Bindings,0,sizeof(Bindings));
    memset(&BindingIpcUdp,0,sizeof(BindingIpcUdp));

    dp = NULL;
    num_target_devices = 0;
    for (dp = dev_base; dp != NULL; dp = dp->next) {
        if (strcmp(LINUX_VPN_IFNAME, dp->name) == 0) {
            continue;
        }
        if (strcmp("lo", dp->name) == 0) {
            /*bind to loopback so we can inject UDP packets
             *for IPC, but don't change the xmit function
             */
            BindingIpcUdp.pDevice = dp;
            BindingIpcUdp.InjectSend = dp->hard_start_xmit;
            BindingIpcUdp.InjectReceive = original_ip_handler.orig_handler_func;
            Bindings[num_target_devices].pPT = original_ip_handler.pt;
            continue;
        }

        /* Don't handle non-eth non-ppp devices */
        if(!supported_device(dp)) {
            continue;
        }
        if (num_target_devices >= MAX_INTERFACES) {
            printk(KERN_DEBUG "exceeded max network devices (%d) at dev %s",
                   MAX_INTERFACES, dp->name);
            break;
        }

        Bindings[num_target_devices].pDevice = dp;
   
        /* store the original mtu for this device. */ 
        Bindings[num_target_devices].original_mtu = dp->mtu;
        /*replace the original send function with our send function*/
        Bindings[num_target_devices].InjectSend = dp->hard_start_xmit;
        dp->hard_start_xmit = replacement_dev_xmit;

        /*copy in the ip packet handler function and packet type struct*/
        Bindings[num_target_devices].InjectReceive
                = original_ip_handler.orig_handler_func;
        Bindings[num_target_devices].pPT = original_ip_handler.pt;

        num_target_devices++;
    }

    if (num_target_devices == 0) {
        printk(KERN_DEBUG "No network devices detected.");
        error = VPNIFUP_FAILURE;
        goto exit_gracefully;
    }

    /* Reply message */
    MOD_INC_USE_COUNT;
exit_gracefully:
    return error;
}
static int handle_vpndown()
{
    int i;
    int error = VPNIFDOWN_SUCCESS;

    if (!MOD_IN_USE) {
        error = VPNIFDOWN_FAILURE;
        goto exit_gracefully;
    }
    cleanup_frag_queue();
    if (num_target_devices == 0) {
        error = VPNIFDOWN_FAILURE;
        goto exit_gracefully;
    }
    /*restore IP packet handler */
    if (original_ip_handler.pt != NULL) {
        original_ip_handler.pt->func
                 = original_ip_handler.orig_handler_func;
    }
    memset(&original_ip_handler,0,sizeof(original_ip_handler));

    /*restore network devices*/
    for (i = 0; i < num_target_devices; i++) {
        struct device *dp =  Bindings[i].pDevice;
        if (dp) {
            dp->hard_start_xmit = Bindings[i].InjectSend;
            Bindings[i].pDevice = NULL;
        }
    }
    memset(&Bindings,0,sizeof(Bindings));

    if (BindingIpcUdp.dst) {
        dst_release(BindingIpcUdp.dst);
        BindingIpcUdp.dst = NULL;
    }
    memset(&BindingIpcUdp,0,sizeof(BindingIpcUdp));
exit_gracefully:
    if (MOD_IN_USE) {
        MOD_DEC_USE_COUNT;
    }
    return error;
}
static void reset_inject_status(inject_status *s)
{
    s->called = FALSE;
    s->rc = 0;
}

int inline ippp_dev(struct device* dev)
{
/* This func does not check for syncPPP
   Check for syncPPP must be done in addition to the checks in these
   functions
*/

    int rc=0;

    if(!dev->name) {
        goto exit_gracefully;
    }

    if(strncmp(dev->name,"ippp",4)) {
        goto exit_gracefully;
    }

    if(dev->name[4]<'0'||dev->name[4]>'9') {
        goto exit_gracefully;
    }

    rc=1;
        
  exit_gracefully:
    return rc;
}

static int inline supported_device(struct device* dev)
{
    int rc=0;

    if(!dev->name) 
    {
        goto exit_gracefully;
    }

    if(!strncmp(dev->name,"eth",3) && (dev->name[3]>='0' && dev->name[3]<='9'))
    {
        rc=1;
    }
    else if(!strncmp(dev->name,"wlan",4) && (dev->name[4]>='0' && dev->name[4]<='9'))
    {
        rc=1;
    }
    else if(!strncmp(dev->name,"ppp",3) && (dev->name[3]>='0' && 
                                            dev->name[3]<='9'))
    {
        rc=1;
    }
    else if(ippp_dev(dev))
    {
        isdn_net_local *lp = (isdn_net_local *) dev->priv;

        if(lp->p_encap==ISDN_NET_ENCAP_SYNCPPP)
        {
            rc=1;
        }
    }

  exit_gracefully:
    return rc;
}

static BINDING *getbindingbydev(struct device * dev)
{
    int i;

    for (i = 0; i <= num_target_devices; i++) {
        if (!strcmp(Bindings[i].pDevice->name, dev->name)) {
            return &Bindings[i];
        }
    }
    return NULL;
}
static CNIFRAGMENT build_ppp_fake_mac_frag(struct ethhdr *dummy)
{
    CNIFRAGMENT MacHdr = NULL;
    
    memset(dummy->h_dest,45,ETH_ALEN);
    memset(dummy->h_source,45,ETH_ALEN);
    dummy->h_proto = htons(ETH_P_IP);

    CniNewFragment(ETH_HLEN,(char*)dummy,&MacHdr, CNI_USE_BUFFER);
    return MacHdr;
}

static int recv_ip_packet_handler(struct sk_buff *skb,
                                    struct device *dev,
                                    struct packet_type *type)
{
    int rc2 = 0;
    int tmp_rc = 0;
    CNISTATUS rc = 0;
    CNIPACKET NewPacket = NULL;
    CNIFRAGMENT Fragment = NULL;
    CNIFRAGMENT MacHdr = NULL;
    PVOID lpReceiveContext = NULL;
    ULONG ulFinalPacketSize;
    BINDING *pBinding = NULL;
    struct ethhdr ppp_dummy_buf;


    if (strcmp("lo", dev->name) == 0) {
        /* grab the routing entry for loopback packets, in case we need
         * to send some later*/
        if (!BindingIpcUdp.dst) {
            BindingIpcUdp.dst = dst_clone(skb->dst);
        }
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }

    /* Don't handle non-eth non-ppp packets */
    if(!supported_device(dev)) {
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }

    pBinding = getbindingbydev(dev);

    /* if we don't have a binding, this is a new device that
    *  has been brought up while the tunnel is up. For now,
    *  just pass the packet
    */
    if (!pBinding) {
        static int firsttime = 1;
        if (firsttime) {
            printk(KERN_DEBUG "RECV: new dev %s detected",
                   dev->name);
            firsttime = 0;
        }
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }
    reset_inject_status(&pBinding->recv_stat);

    /*get the mac header. PPP devices need to create a fake mac header*/
    if(ippp_dev(dev))
    {
        isdn_net_local *lp = (isdn_net_local *) dev->priv;
        
        if(lp->p_encap!=ISDN_NET_ENCAP_SYNCPPP)
        {
            rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
            goto exit_gracefully;
        }
        else
        {
            MacHdr = build_ppp_fake_mac_frag(&ppp_dummy_buf);
        }
    }
    else
    {
        switch (skb->dev->hard_header_len)
        {
        case ETH_HLEN:
            CniNewFragment(ETH_HLEN, skb->mac.raw,&MacHdr, CNI_USE_BUFFER);
            break;
        case PPP_HLEN:
            MacHdr = build_ppp_fake_mac_frag(&ppp_dummy_buf);
            break;
        default:
            printk(KERN_DEBUG "unknown mac header length (%d)\n",
                   skb->dev->hard_header_len);
            dev_kfree_skb(skb);
            skb = NULL;
            goto exit_gracefully;
        }
    }

    CniNewFragment(skb->len, skb->data, &Fragment, CNI_USE_BUFFER);
    ulFinalPacketSize = skb->len;

    rc = CNICallbackTable.Receive((void *) &pBinding,
                                  &lpReceiveContext,
                                  &MacHdr,
                                  &Fragment,
                                  &NewPacket, &ulFinalPacketSize);

    switch (rc) {
    case CNI_CONSUME:
        tmp_rc = CNICallbackTable.ReceiveComplete(pBinding,
                                                  lpReceiveContext,
                                                  NewPacket);
        dev_kfree_skb(skb);

        if (pBinding->recv_stat.called) {
            rc2 = pBinding->recv_stat.rc;
            rx_packetcount++;
        }
        break;
    case CNI_CHAIN:
        tmp_rc = CNICallbackTable.ReceiveComplete(pBinding,
                                                  lpReceiveContext,
                                                  NewPacket);

        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);

        if (pBinding->recv_stat.called) {
            rc2 = pBinding->recv_stat.rc;
        }

        rx_packetcount++;
        break;
    case CNI_DISCARD:
        dev_kfree_skb(skb);
        break;
    default:
        printk(KERN_DEBUG "RECV: Unhandled case in %s rc was %x",
               __FUNCTION__, (uint)rc);

        dev_kfree_skb(skb);
    }
exit_gracefully:    
    if (MacHdr)
    {
        CniReleaseFragment(MacHdr, CNI_KEEP_BUFFERS);
    }
    if (Fragment)
    {
        CniReleaseFragment(Fragment, CNI_KEEP_BUFFERS);
    }
    return rc2;
}

static int do_cni_send(BINDING* pBinding,struct sk_buff *skb,
                        struct device *dev)
{
    int rc2 = 0;
    CNISTATUS rc = 0;
    CNIPACKET Packet = NULL;
    CNIFRAGMENT Fragment = NULL;
    CNIFRAGMENT MacHdr = NULL;
    PVOID lpSendContext = NULL;
    struct ethhdr ppp_dummy_buf;
    int hard_header_len = 0;

    reset_inject_status(&pBinding->send_stat);

    /*get the mac header. PPP devices need to create a fake mac header*/
    if(ippp_dev(dev))
    {
        isdn_net_local *lp = (isdn_net_local *) dev->priv;
        
        if(lp->p_encap!=ISDN_NET_ENCAP_SYNCPPP)
        {
            /* This part should never be reached */
            rc2 = pBinding->InjectSend(skb, dev);
            goto exit_gracefully;
        }
        else
        {
            MacHdr = build_ppp_fake_mac_frag(&ppp_dummy_buf);
            hard_header_len = IPPP_MAX_HEADER;
        }
    }
    else
    {
        switch (skb->dev->hard_header_len)
        {
        case ETH_HLEN:
            CniNewFragment(ETH_HLEN, skb->data,
                           &MacHdr, CNI_USE_BUFFER);
            hard_header_len = ETH_HLEN;
            break;
        case PPP_HLEN:
            MacHdr = build_ppp_fake_mac_frag(&ppp_dummy_buf);
            /* note: the PPP device says it's hard_header_len is 4,
             * but skb->data points at the IP header*/
            hard_header_len = 0;
            break;
        default:
            printk(KERN_DEBUG "unknown mac header length (%d)\n",
                   skb->dev->hard_header_len);
            dev_kfree_skb(skb);
            skb = NULL;
            goto exit_gracefully;
        }
    }

    CniNewPacket(0, &Packet);
    /*skb->data points to the mac header, the fragment should start
     *with the ip header */
    CniNewFragment(skb->len  - hard_header_len,
                   skb->data + hard_header_len, &Fragment,
                   CNI_USE_BUFFER);

    CniAddFragToFront(Packet, Fragment);

    rc = CNICallbackTable.Send((void *) &pBinding,
                               &lpSendContext, &MacHdr, &Packet);

    switch (rc) {
    case CNI_DISCARD:
        /* packet was tunneled */
        if (pBinding->send_stat.called)
        {
            rc2 = pBinding->send_stat.rc;

            /*if the packet was tunneled, rc2 should
              now contain the return code from the
              call to the nic's hard_start_xmit function.
              if that function failed, the kernel is going
              to queue this skb and send it to us again later,
              so don't free it. */
            if (rc2 == 0)
            {
                dev_kfree_skb(skb);
            }
        }
        /* packet dropped*/
        else
        {
            dev_kfree_skb(skb);
        }
        break;
    case CNI_CHAIN:
        rc2 = pBinding->InjectSend(skb, dev);
        tx_packetcount++;
        break;
    default:
        printk(KERN_DEBUG "Unhandled case in %s rc was %x", __FUNCTION__,
               (uint)rc);

        dev_kfree_skb(skb);
        rc2 = 0;
    }
exit_gracefully:
    if (MacHdr)
    {
        CniReleaseFragment(MacHdr, CNI_KEEP_BUFFERS);
    }
    if (Packet)
    {
        CniReleasePacket(Packet, CNI_KEEP_BUFFERS);
    }
    return rc2;
}
static int replacement_dev_xmit(struct sk_buff *skb,
                                            struct device *dev)
{
    int rc2 = 0;
    BINDING *pBinding = 0;

    pBinding = getbindingbydev(dev);
    /* if we don't have a binding, this is a new device that
    *  has been brought up while the tunnel is up. For now,
    *  just pass the packet
    */
    if (!pBinding) {
        static int firsttime = 1;
        if (firsttime) {
            printk(KERN_DEBUG "XMIT: new dev %s detected",
                   dev->name);
            firsttime = 0;
        }
                           
        rc2 = dev->hard_start_xmit(skb, dev);
        goto exit_gracefully;
    }
    if (need_reorder_frag(skb))
    {
        rc2 = handle_fragment(pBinding,skb,dev);
    }
    else
    {
        rc2 = do_cni_send(pBinding,skb,dev);
    }
exit_gracefully:
    return rc2;
}
