/**************************************************************************
*           Copyright (c) 2001, Cisco Systems, All Rights Reserved
***************************************************************************
*
*  File:    linuxcniapi.c
*  Date:    22/03/01
*
***************************************************************************
* This module implements a translation layer between the CNI API and the
* Linux Interceptor driver.
***************************************************************************/

#include <linux/version.h>
#include <linux/config.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/netdevice.h>
#include <linux/if.h>
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/vmalloc.h>
#include <net/dst.h>

#include "linux_os.h"
#include    "Cniapi.h"
#include "linuxcniapi.h"
/******************************** Globals *********************************/

CNI_CHARACTERISTICS CNICallbackTable; /* This stores the Plugin's function pointers */
PCHAR pcDeviceName;             /* Ignore. Only our pluggin so we don't care about it */

#define CNI_SIGNATURE		"@CNI"
#define CNI_SIGNATURESIZE	4

typedef struct {
    char Signature[CNI_SIGNATURESIZE]; // not a null terminated string!  
    UINT blockSize;
} BLOCKINFO, *LPBLOCKINFO;


typedef struct {
    PVOID lpFragmentData;
    UINT uiFragmentDataSize;
    struct FRAGMENTBUFFER *lpPrevious;
    struct FRAGMENTBUFFER *lpNext;
} FRAGMENTBUFFER, *LPFRAGMENTBUFFER;


typedef struct {
    char Signature[CNI_SIGNATURESIZE];

    UINT uiPacketSize;
    UINT uiFragmentCount;

    LPFRAGMENTBUFFER lpHead;
    LPFRAGMENTBUFFER lpTail;
} PACKETDESCRIPTOR, *LPPACKETDESCRIPTOR;

extern BINDING *BindingIpcUdp;
extern int inline ippp_dev(struct device* dev);
/********************************* Local private function ***********************/

/* allociate fragment buffer*/
CNISTATUS INTER_CNI_Allocate_Buffer(UINT uiLength,
                                    PCHAR pcBuffer,
                                    PVOID * ppFragmentBuffer, UINT uiflags)
{
    CNISTATUS Status;
    LPFRAGMENTBUFFER lpFragmentBuffer = NULL;
    PVOID pVirtualAddress = NULL;

    if (!ppFragmentBuffer)
        return CNI_E_BAD_PARAMETER;

    if ((!pcBuffer) || (uiflags == CNI_COPY_BUFFER)) {
        if ((Status = CNI_LINUXMemAlloc(uiLength, &pVirtualAddress))
            != CNI_SUCCESS)
            return (Status);
    } else
        pVirtualAddress = (PVOID) pcBuffer; // CNI_USE_BUFFER

    lpFragmentBuffer =
        (LPFRAGMENTBUFFER) kmalloc(sizeof(FRAGMENTBUFFER), GFP_ATOMIC);

    if (lpFragmentBuffer) {
        memset(lpFragmentBuffer, 0, sizeof(FRAGMENTBUFFER));
        lpFragmentBuffer->lpFragmentData = pVirtualAddress;
        lpFragmentBuffer->uiFragmentDataSize = uiLength;
        lpFragmentBuffer->lpPrevious = NULL;
        lpFragmentBuffer->lpNext = NULL;
    } else {
        if ((!pcBuffer) || (uiflags == CNI_COPY_BUFFER))
            CNI_LINUXMemFree(pVirtualAddress); // Currently always passes
        return CNI_W_OUT_OF_DESCRIPTORS;
    }

    /* If pcBuffer == NULL or CNI_USE_BUFFER, we are done. 
       For CNI_COPY_BUFFER, we have to copy the data into the new buffer */

    if (uiflags == CNI_COPY_BUFFER)
        memcpy(pVirtualAddress, pcBuffer, uiLength);

    *ppFragmentBuffer = lpFragmentBuffer;
    return CNI_SUCCESS;
}

/***********************************************************************
 *
 * Deallocate a buffer.
 *
 ***********************************************************************/

CNISTATUS INTER_CNI_Deallocate_Buffer(PVOID pFragmentBuffer, UINT uiFlags)
{
    LPFRAGMENTBUFFER lpFragmentBuffer = NULL;

    if (!pFragmentBuffer)
        return CNI_SUCCESS;

    if (uiFlags == CNI_RELEASE_BUFFERS) {
        lpFragmentBuffer = (LPFRAGMENTBUFFER) pFragmentBuffer;

        CNI_LINUXMemFree(lpFragmentBuffer->lpFragmentData);
    }

    kfree(pFragmentBuffer);

    return CNI_SUCCESS;
}

void QueryBuffer(LPFRAGMENTBUFFER lpFragment,
                 PVOID * lpVirtualAddress, UINT * lpuiSize)
{
    UINT blockSize = 0xffffffff;

    LPBLOCKINFO lpBlockInfo =
        (LPBLOCKINFO) ((UCHAR *) lpFragment->lpFragmentData -
                       sizeof(BLOCKINFO));

    if (memcmp(lpBlockInfo->Signature, CNI_SIGNATURE, CNI_SIGNATURESIZE) ==
        0) blockSize = lpBlockInfo->blockSize;

    if (lpVirtualAddress)
        *lpVirtualAddress = lpFragment->lpFragmentData;

    if (lpuiSize)
        *lpuiSize = blockSize;
}

/*******************************************************************************/








/*************************************************************************
* 
* CNI_LINUXMemAlloc
* 
* Description:
* This allocates uiSize bytes of memory and stores a pointer to this memory in the location pointed to by ppMem.
* 
*	
* Argument:
* 
*    IN UINT uiSize - an unsigned integer that contains the 
*                              number of bytes to allocate.
*   OUT CHAR **ppMem - points to a memory pointer that will
*                               contain the allocated memory.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - Allocation succeeded.
*                CNI_W_OUT_OF_RESOURCES - Unable to allocate memory due 
*                to lack of resources.
*                CNI_E_BAD_PARAMETER - The pMem pointer was NULL.
* 
*************************************************************************/
CNISTATUS CNI_LINUXMemAlloc(IN UINT uiSize, OUT PVOID * ppMem)
{
    LPBLOCKINFO lpBlockInfo;

    if (ppMem == NULL)
        return CNI_E_BAD_PARAMETER;

    *ppMem = kmalloc(uiSize + sizeof(BLOCKINFO), GFP_ATOMIC);

    if (*ppMem) {
        memset(*ppMem, 0, uiSize + sizeof(BLOCKINFO));
        lpBlockInfo = (LPBLOCKINFO) * ppMem;

        ((UCHAR *) * ppMem) += sizeof(BLOCKINFO);

        /*put the signature at the beginning of the block */
        memcpy(lpBlockInfo->Signature, CNI_SIGNATURE, CNI_SIGNATURESIZE);

        /*put the block side after the signature */
        lpBlockInfo->blockSize = uiSize;

        return CNI_SUCCESS;
    }

    return CNI_W_OUT_OF_RESOURCES;
}

/*************************************************************************
* 
* CNI_LINUXMemRealloc
* 
* Description:
* This allocates allocates uiSize bytes of memory and copies the data from
* pMem to it.  It then frees pMem, and returns the newly allocated buffer.
* If memory was not able to be allocated then pMem is NOT freed and this
* call returns a NULL pointer. Note that if uiSize is 0 this function will
* do nothing.
* 
*	
* Argument:
* 
*    IN UINT uiSize - an unsigned integer that contains the 
*                              number of bytes to allocate.  Zero will result
*                              in this function doing nothing and returning
*                              NULL.
*   OUT PVOID pMem  - A pointer that was previously returned by a call to
*                              CniMemAlloc().
* 
* Returns:
* 
*   PVOID      - NULL - the allocation did not succeed, pMem was not freed.
*                Anything else - the allocation succeeded and the the data
*                at pMem was copied to 
*
***************************************************************************/
PVOID CNI_LINUXMemRealloc(IN UINT uiSize, IN PVOID pMem)
{
    LPBLOCKINFO lpBlockInfo;
    UINT uiOldBlockSize;
    PVOID pNewMem = NULL;

    if (!pMem)
        return NULL;

    lpBlockInfo = (LPBLOCKINFO) ((UCHAR *) pMem - sizeof(BLOCKINFO));

    /*Check the signature */
    if (memcmp(lpBlockInfo->Signature, CNI_SIGNATURE, CNI_SIGNATURESIZE) !=
        0) return NULL;         // this block is not allocated by CNI_LINUXalloc!

    uiOldBlockSize = lpBlockInfo->blockSize;

    if (CNI_LINUXMemAlloc(uiSize, &pNewMem) == CNI_SUCCESS) {
        /*make sure we are not over write the new block */
        if (uiOldBlockSize > uiSize)
            uiOldBlockSize = uiSize;

        memcpy(pNewMem, pMem, uiOldBlockSize);

        CNI_LINUXMemFree(pMem);
    }

    return pNewMem;
}

/*************************************************************************
* 
* CNI_LINUXMemFree
* 
* Description:
* This frees memory that has been allocated by a call to CniMemAlloc()
*	
* 
* Argument:
* 
*   IN CHAR *pMem - Pointer to the memory that is to be freed.
* 
* Returns:
* 
*   CNISTATUS  -  CNI_SUCCESS - The memory was freed.
*	              CNI_E_BAD_PARAMETER - The pMem pointer was NULL.
*	              CNI_E_BAD_MEMORY - The memory was not allocated by this plugin.
*************************************************************************/
CNISTATUS CNI_LINUXMemFree(IN CHAR * pMem)
{
    LPBLOCKINFO lpBlockInfo;

    if (!pMem)
        return CNI_E_BAD_PARAMETER;

    lpBlockInfo = (LPBLOCKINFO) (pMem - sizeof(BLOCKINFO));

    /*Check the signature */
    if (memcmp(lpBlockInfo->Signature, CNI_SIGNATURE, CNI_SIGNATURESIZE) !=
        0) return CNI_E_BAD_MEMORY;

    //Clear the signature.
    memset(lpBlockInfo->Signature, 0, CNI_SIGNATURESIZE);

    // free the real block address pointer!
    kfree(lpBlockInfo);

    return CNI_SUCCESS;

}



/*************************************************************************
* 
* CNI_LINUXNewPacket
* 
* Description:
* This allocates a new packet structure with a data area the of uiSize 
* bytes.  The packet structure pointed to by pPacket may be larger or 
* smaller than uiSize bytes depending on the operating system this API is
* implemented on.  Only the CNI packet functions should be used to access
* the data stored in pPacket.
*
* 
*	
* Argument:
* 
*   IN UINT uiSize - The size in bytes of the packets data.
*   IN OUT PCNIPACKET pPacket - A pointer to a CNI Packet data
*                                        structure.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - Allocation succeeded.
*                CNI_W_OUT_OF_DESCRIPTORS - All the CNIPACKET descriptors
*                have been allocated.
*                CNI_W_OUT_OF_RESOURCES - Low on memory.
*                CNI_E_BAD_PARAMETER - The pPacket pointer was NULL.
*************************************************************************/
CNISTATUS CNI_LINUXNewPacket(IN UINT uiSize, IN OUT PCNIPACKET pPacket)
{
    CNISTATUS rc;
    LPPACKETDESCRIPTOR lpPacketDescriptor;

    if (!pPacket)
        return CNI_E_BAD_PACKET;

    lpPacketDescriptor =
        (LPPACKETDESCRIPTOR) kmalloc(sizeof(PACKETDESCRIPTOR), GFP_ATOMIC);

    if (!lpPacketDescriptor) {
        return CNI_W_OUT_OF_DESCRIPTORS;
    }
    memset(lpPacketDescriptor, 0, sizeof(PACKETDESCRIPTOR));
    memcpy(lpPacketDescriptor->Signature, CNI_SIGNATURE,
           CNI_SIGNATURESIZE);

    lpPacketDescriptor->uiPacketSize = uiSize;

    if (uiSize > 0) {
        lpPacketDescriptor->uiFragmentCount = 1;
        rc = CNI_LINUXNewFragment(uiSize,
                                  NULL,
                                  (CNIFRAGMENT *) & (lpPacketDescriptor->
                                                     lpHead),
                                  CNI_USE_BUFFER);

        if (rc != CNI_SUCCESS) {
            kfree(lpPacketDescriptor);
            return CNI_W_OUT_OF_RESOURCES;
        }
    } else {
        lpPacketDescriptor->uiFragmentCount = 0;
        lpPacketDescriptor->lpHead = NULL;
    }

    lpPacketDescriptor->lpTail = lpPacketDescriptor->lpHead;

    *pPacket = (PCNIPACKET) lpPacketDescriptor;

    return CNI_SUCCESS;
}



/*************************************************************************
* 
* CNI_LINUXReleasePacket
* 
* Description:
* This function will free packets allocated by the plugin via a 
* CniAllocatePacket().  The ulFlags parameter, when set to CNI_KEEP_BUFFERS
* indicates that the memory buffers associated with this packet will not
* be freed by CNI.  Otherwise, CniReleasePacket() will first call
* CniMemFree() for all of the memory buffers associated with Packet
* before freeing the packet.  In this case all memory buffers associated
* with a packet must have been allocated via CniMemAllocate(), otherwise
* the result of this call is undefined.
*	
* Argument:
* 
*   IN CNIPACKET Packet - A packet structure to be released.
*   IN ULONG     ulFlags  - CNI_KEEP_BUFFERS or nothing to free them.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - Allocation succeeded.
*                CNI_E_BAD_PARAMETER - The Packet parameter was not valid.
*                CNI_E_BAD_PACKET - This Packet was not allocated by this
*                plugin.
* 
*************************************************************************/
CNISTATUS CNI_LINUXReleasePacket(IN CNIPACKET Packet, IN ULONG ulFlags)
{
    LPPACKETDESCRIPTOR lpPacketDescriptor;
    LPFRAGMENTBUFFER lpFragmentBuffer;
    LPFRAGMENTBUFFER lpNextFragmentBuffer;

    if (!Packet)
        return CNI_E_BAD_PARAMETER;

    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;

    if (memcmp
        (lpPacketDescriptor->Signature, CNI_SIGNATURE,
         CNI_SIGNATURESIZE) != 0)
        return CNI_E_BAD_PACKET;

    lpFragmentBuffer = lpPacketDescriptor->lpHead;

    while (lpFragmentBuffer) {
        lpNextFragmentBuffer = (LPFRAGMENTBUFFER) lpFragmentBuffer->lpNext;
        INTER_CNI_Deallocate_Buffer(lpFragmentBuffer, ulFlags);
        lpFragmentBuffer = lpNextFragmentBuffer;
    }

    kfree(Packet);

    return CNI_SUCCESS;
}



/*************************************************************************
* 
* CNI_LINUXGetPacketData
* 
* Description:
* This function will copy the ulSize bytes of packet data, starting from
* ulOffset, to the buffer pointed to by *pBuffer.
* 
* 
* Argument:
*	
*   IN CNIPACKET Packet - The packet the data will be retrieved from.
*   IN ULONG ulOffset - The byte-offset from the beginning of the
*                                packet data from which to start copying the
*                                data.
*   IN ULONG ulSize - The number of bytes to retrieve from the packet
*   OUT PCHAR pBuffer - A character pointer that contains at least enough
*                                 storage to hold the ulSize bytes of data.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The data was retrieved.
*                CNI_W_NEED_BUFFER - The pBuffer was NULL.
*                CNI_W_PACKET_TOO_SMALL - ulOffset plus ulSize exceed the
                                          end of the packet.
*                CNI_W_OUT_OF_RANGE - ulOffset was past the end of the packet.
*                CNI_E_BAD_PACKET - The Packet was not valid.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXGetPacketData(IN CNIPACKET Packet,
                       IN ULONG ulOffset,
                       IN ULONG ulSize, OUT PCHAR pBuffer)
{
    ULONG ulFragmentIndex;
    ULONG ulDataIndex = 0;
    LPFRAGMENTBUFFER lpFragment;
    LPPACKETDESCRIPTOR lpPacketDescriptor;
    PCHAR lpStart;
    ULONG ulBytes;
    ULONG ulLocalOffset;

    if (!Packet)
        return CNI_E_BAD_PACKET;

    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;

    if (memcmp
        (lpPacketDescriptor->Signature, CNI_SIGNATURE,
         CNI_SIGNATURESIZE) != 0)
        return CNI_E_BAD_PACKET;

    if ((ulOffset + ulSize) > lpPacketDescriptor->uiPacketSize)
        return CNI_W_PACKET_TOO_SMALL;

    if (!pBuffer)
        return CNI_W_NEED_BUFFER;

    for (ulFragmentIndex = 0, lpFragment =
         (LPFRAGMENTBUFFER) lpPacketDescriptor->lpHead;
         ulFragmentIndex < lpPacketDescriptor->uiFragmentCount
         && ulSize != 0;
         ulFragmentIndex++, lpFragment =
         (LPFRAGMENTBUFFER) lpFragment->lpNext) {
        ulDataIndex += lpFragment->uiFragmentDataSize;
        if (ulDataIndex > ulOffset) {
            /* some or all of the data are in this fragment */
            ulLocalOffset =
                lpFragment->uiFragmentDataSize - (ulDataIndex - ulOffset);

            lpStart = (UCHAR *) lpFragment->lpFragmentData + ulLocalOffset;

            ulBytes = lpFragment->uiFragmentDataSize - ulLocalOffset;

            /* check to see if the last bit of data end in the fragment */
            if (ulBytes > ulSize)
                ulBytes = ulSize;

            /* updata the offset to next location */
            ulOffset += ulBytes;
            /* reduce the size since we updated the offset */
            ulSize -= ulBytes;

            /* Copy some or all of the data into the desination buffer */
            memcpy(pBuffer, lpStart, ulBytes);

            /* move pBuffer pointer to the end of the data */
            pBuffer += ulBytes;

        }
    }

    return CNI_SUCCESS;
}



/*************************************************************************
* 
* CNI_LINUXSetPacketData
* 
* Description:
* This function will copy the ulSize bytes from pBuffer to Packet at the
* data offset of ulOffset.  If ulFlags has CNI_TRUNCATE_PACKET set
* CniSetPacketData() will truncate the packet at the end of the copied
* data by adjusting the packet and fragment size.
* 
* 
* Argument:
* 
*   IN CNIPACKET Packet - The packet the data will be copied into.
*   IN ULONG ulOffset - The byte-offset from the beginning of the 
*                                packet to start copying to.
*   IN ULONG ulSize - The number of bytes that will be set to the
*                              packet data.
*   IN PCHAR pBuffer - Points to a buffer that contains the data 
*                               to set to the packet.
*   IN ULONG ulFlags - CNI_TRUNCATE_PACKET or nothing to not truncate.
*
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The data was set.
*                CNI_W_PACKET_TOO_SMALL - ulOffset plus ulSize exceed the
*                end of the packet.
*                CNI_W_COULD_NOT_SHORTEN - The packet could not be truncated.
*                CNI_W_OUT_OF_RANGE - ulOffset exceeds the end of the packet.
*                CNI_E_BAD_PARAMETER - pBuffer was NULL.
*                CNI_E_BAD_PACKET - The Packet was not valid.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXSetPacketData(IN CNIPACKET Packet,
                       IN ULONG ulOffset,
                       IN ULONG ulSize, IN PCHAR pBuffer, IN ULONG ulFlags)
{
    ULONG ulFragmentIndex;
    ULONG ulDataIndex = 0;
    LPFRAGMENTBUFFER lpFragment = NULL;
    LPFRAGMENTBUFFER lpLastFragment = NULL;
    LPFRAGMENTBUFFER lpNextFragment = NULL;
    LPPACKETDESCRIPTOR lpPacketDescriptor = NULL;
    PCHAR lpStart;
    ULONG ulBytes=0;
    ULONG ulLocalOffset=0;

    if (!Packet)
        return CNI_E_BAD_PACKET;

    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;

    if (memcmp
        (lpPacketDescriptor->Signature, CNI_SIGNATURE,
         CNI_SIGNATURESIZE) != 0)
        return CNI_E_BAD_PACKET;

    if ((ulOffset + ulSize) > lpPacketDescriptor->uiPacketSize)
        return CNI_W_PACKET_TOO_SMALL;

    if (!pBuffer)
        return CNI_W_NEED_BUFFER;

    for (ulFragmentIndex = 0, lpFragment =
         (LPFRAGMENTBUFFER) lpPacketDescriptor->lpHead;
         ulFragmentIndex < lpPacketDescriptor->uiFragmentCount
         && ulSize != 0;
         ulFragmentIndex++, lpFragment =
         (LPFRAGMENTBUFFER) lpFragment->lpNext) {
        ulDataIndex += lpFragment->uiFragmentDataSize;
        if (ulDataIndex > ulOffset) {
            /* some or all of the data are in this fragment */
            ulLocalOffset =
                lpFragment->uiFragmentDataSize - (ulDataIndex - ulOffset);

            lpStart = (UCHAR *) lpFragment->lpFragmentData + ulLocalOffset;

            ulBytes = lpFragment->uiFragmentDataSize - ulLocalOffset;

            /* check to see if the last bit of data end in the fragment */
            if (ulBytes > ulSize)
                ulBytes = ulSize;

            /* updata the offset to next location */
            ulOffset += ulBytes;
            /* reduce the size since we updated the offset */
            ulSize -= ulBytes;

            /* Copy some or all of the data into the desination buffer */
            memcpy(lpStart, pBuffer, ulBytes);

            /* move pBuffer pointer to the end of the data */
            pBuffer += ulBytes;

            lpLastFragment = lpFragment;
        }

    }


    /* truncate the packet if needed */
    if (ulSize == 0 && ulFlags == CNI_TRUNCATE_PACKET) {
        UINT ulTruncateSize = lpLastFragment->uiFragmentDataSize;
        UINT ulNumTruncateFragments = 0;

        lpLastFragment->uiFragmentDataSize = ulBytes + ulLocalOffset;

        ulTruncateSize -= lpLastFragment->uiFragmentDataSize;

        lpFragment = (LPFRAGMENTBUFFER) lpLastFragment->lpNext;
        lpLastFragment->lpNext = NULL;

        lpPacketDescriptor->lpTail = lpLastFragment;

        while (lpFragment) {
            ulNumTruncateFragments++;
            ulTruncateSize += lpFragment->uiFragmentDataSize;

            lpNextFragment = (LPFRAGMENTBUFFER) lpFragment->lpNext;
            INTER_CNI_Deallocate_Buffer(lpFragment, CNI_RELEASE_BUFFERS);
            lpFragment = lpNextFragment;
        }

        lpPacketDescriptor->uiPacketSize -= ulTruncateSize;
        lpPacketDescriptor->uiFragmentCount -= ulNumTruncateFragments;
    }

    return CNI_SUCCESS;
}




/*************************************************************************
* 
* CNI_LINUXQueryPacket
* 
* Description:
* This function will query the Packet to determine it's size, number of 
* fragments and first and last fragments.  If any of the OUT pointers are 
* NULL that value will not be returned.  This allows a driver writer to
* only retrieve the information they are interested in.
*
* 
* 
* Argument:
* 
*   IN CNIPACKET Packet -	The packet to query.
*   OUT PULONG pulSize OPTIONAL - The number of data bytes in 
*                the packet, this may be NULL.
*   OUT PULONG pulNumFragments OPTIONAL - The number of fragments
*                that make up a the packet, this may be NULL.
*   OUT PCNIFRAGMENT pFirstFragment OPTIONAL - The first data 
*                fragment in the packet, this may be NULL.
*   OUT PCNIFRAGMENT pLastFragment OPTIONAL - The last data 
*                fragment in the packet, this may be NULL.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - Packet info was retrieved.
*                CNI_E_BAD_PARAMETER - All parameters were NULL.
*                CNI_E_BAD_PACKET - The Packet was not valid.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXQueryPacket(IN CNIPACKET Packet,
                     OUT PULONG pulSize OPTIONAL,
                     OUT PULONG pulNumFragments OPTIONAL,
                     OUT PCNIFRAGMENT pFirstFragment OPTIONAL,
                     OUT PCNIFRAGMENT pLastFragment OPTIONAL)
{
    LPPACKETDESCRIPTOR lpPacketDescriptor;

    if (!Packet)
        return CNI_E_BAD_PACKET;

    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;

    if (!pulSize && !pulNumFragments && !pFirstFragment && !pLastFragment)
        return CNI_E_BAD_PARAMETER;

    if (pulSize)
        *pulSize = lpPacketDescriptor->uiPacketSize;

    if (pulNumFragments)
        *pulNumFragments = lpPacketDescriptor->uiFragmentCount;

    if (pFirstFragment)
        *pFirstFragment = lpPacketDescriptor->lpHead;

    if (pLastFragment)
        *pLastFragment = lpPacketDescriptor->lpTail;

    return CNI_SUCCESS;
}



/*************************************************************************
* 
* CNI_LINUXAddFragToFront
* 
* Description:
* This function adds a new Fragment to the front of Packet's fragment chain.
* 
*	
* Argument:
* 
*   IN CNIPACKET Packet - The packet to which the fragment will 
*               be added.
*   IN CNIFRAGMENT Fragment - The fragment that will be added to 
*               the packet.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - Packet info was retrieved.
*                CNI_E_BAD_FRAGMENT - The Fragment was not valid.
*                CNI_E_BAD_PACKET - The Packet was not valid.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXAddFragToFront(IN CNIPACKET Packet, IN CNIFRAGMENT Fragment)
{
    LPFRAGMENTBUFFER lpFragmentBuffer;
    LPPACKETDESCRIPTOR lpPacketDescriptor;

    if (!Packet)
        return CNI_E_BAD_PACKET;

    if (!Fragment)
        return CNI_E_BAD_FRAGMENT;


    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;
    lpFragmentBuffer = (LPFRAGMENTBUFFER) Fragment;


    // update descriptor data
    lpPacketDescriptor->uiFragmentCount++;
    lpPacketDescriptor->uiPacketSize +=
        lpFragmentBuffer->uiFragmentDataSize;


    // this is the head fragement so therfore Previous fragment must be null
    lpFragmentBuffer->lpPrevious = (struct FRAGMENTBUFFER *) NULL;
    lpFragmentBuffer->lpNext =
        (struct FRAGMENTBUFFER *) lpPacketDescriptor->lpHead;

    if (lpPacketDescriptor->lpHead)
        lpPacketDescriptor->lpHead->lpPrevious =
            (struct FRAGMENTBUFFER *) lpFragmentBuffer;

    lpPacketDescriptor->lpHead = (LPFRAGMENTBUFFER) lpFragmentBuffer;

    if (!lpPacketDescriptor->lpTail)
        lpPacketDescriptor->lpTail = lpFragmentBuffer;

    return CNI_SUCCESS;
}


/*************************************************************************
* 
* CNI_LINUXCopyFragment
* 
* Description:
* This function copies the fragment structure of SourceFragment to the
* allocated pDestinationFragment structure.  If ulFlags has CNI_COPY_BUFFER
* set then a new data buffer will be created for the destination fragment,
* pDestFragment will point to the same data buffer as SourceFragment. 
* If a fragment is copied without copying the data buffer, then the copy
* must be freed before the source, and the copies data buffer must NOT be
* freed (since it belongs to the source).
* 
* Argument:	
* 
*   IN CNIFRAGMENT SourceFragment - The fragment that will be copied.
*   IN OUT PCNIFRAGMENT pDestFragment - Points to the fragment 
*                structure that will be allocated and on behalf of the 
*                plugin and filled with the SourceFragment info.
*   IN ULONG ulFlags - CNI_COPY_BUFFER or nothing to use the source
*                fragments buffer pointer.* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The SourceFragment was copied.
*                CNI_W_FRAG_DESC_FAILURE - All fragments have been 
*                allocated, there are no more available.
*                CNI_W_BAD_FORMAT - The SourceFragment is not valid.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXCopyFragment(IN CNIFRAGMENT SourceFragment,
                      IN OUT PCNIFRAGMENT pDestFragment, IN ULONG ulFlags)
{
    CNISTATUS Status;

    LPFRAGMENTBUFFER lpFragmentBuffer = (LPFRAGMENTBUFFER) SourceFragment;


    if (ulFlags == CNI_COPY_BUFFER)
        Status =
            INTER_CNI_Allocate_Buffer(lpFragmentBuffer->uiFragmentDataSize,
                                      lpFragmentBuffer->lpFragmentData,
                                      (PVOID *) pDestFragment,
                                      CNI_COPY_BUFFER);
    else
        Status =
            INTER_CNI_Allocate_Buffer(lpFragmentBuffer->uiFragmentDataSize,
                                      lpFragmentBuffer->lpFragmentData,
                                      (PVOID *) pDestFragment,
                                      CNI_USE_BUFFER);

    return Status;
}



/*************************************************************************
* 
* CNI_LINUXNewFragment
* 
* Description:
* This function creates packet fragment and sets the fragment structure.
* The data used will be taken from pBuffer, if pBuffer is NULL ulFlags
* will be ignored and a new buffer of size ulSize will be allocated with
* CniMemAlloc().  Otherwise if ulFlags is set to CNI_COPY_BUFFER, a new
* buffer will be allocated and *pBuffer will be copied to it.  If ulFlags
* is CNI_USE_BUFFER no buffer will be allocated and pBuffer will be used
* for the Fragment data. 
* 
* Argument:
*	
*   IN ULONG ulSize - The size, in bytes, of the data pointed to
*               by pBuffer.
*   IN CHAR *pBuffer - Points to a buffer that will be used as 
*               the packet data.
*   OUT CNIFRAGMENT *pFragment - A pointer to a new fragment 
*               structure.
*   IN ULONG ulFlags - CNI_USE_BUFFER or CNI_COPY_BUFFER
*
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The Fragment was created successfuly
*                CNI_E_OUT_OF_MEMORY - Failed to allocate memory for fragment
*                CNI_W_FRAG_DESC_FAILURE - Exceeded the maximum number of fragments allocated.
*                CNI_E_BAD_PARAMETER - The pFragment pointer was invalid.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXNewFragment(IN ULONG ulSize,
                     IN CHAR * pBuffer,
                     OUT CNIFRAGMENT * pFragment, IN ULONG ulFlags)
{
    return INTER_CNI_Allocate_Buffer(ulSize, pBuffer,
                                     (CNIFRAGMENT *) pFragment, ulFlags);
}


/*************************************************************************
* 
* CNI_LINUXReleaseFragment
* 
* Description:
* This releases a fragment and optionally the buffer associated with that
* fragment.  ulFlags can be set to CNI_KEEP_BUFFERS to not release the
* buffers in use by the fragment or it can be set to CNI_RELEASE_BUFFERS
* so that the buffers in use by the fragment will be released.
*	
* Argument:
* 
*   IN CNIFRAGMENT Fragment - A packet fragment to be released.
*   IN ULONG ulFlags - CNI_KEEP_BUFFERS or CNI_RELEASE_BUFFERS* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The Fragment was created successfuly
*                CNI_E_BAD_PARAMETER - The Fragment was invalid.
*                CNI_E_BAD_FRAGMENT - The Fragment being released was not
*                allocated by this plugin.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXReleaseFragment(IN CNIFRAGMENT Fragment, IN ULONG ulFlags)
{
    return (INTER_CNI_Deallocate_Buffer(Fragment, ulFlags));
}


/*************************************************************************
* 
* CNI_LINUXSetFragmentLength
* 
* Description:
* This changes a fragment's length.  If that fragment is part of a packet 
* then the packet should be passed as well.  If the fragment is not a part
* of a packet then Packet should be NULL.  A fragment can not be made 
* larger than the size that was originally allocated for it.
* 
* 
* Argument:	
* 
*   IN OUT CNIFRAGMENT Fragment - A packet fragment to be released.
*   IN ULONG ulLength - The fragment length to be set.
*   IN OUT CNIPACKET Packet - The packet that Fragment is a part of.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The Fragment length was changed.
*                CNI_E_BAD_FRAGMENT - The Fragment is not a CNIFRAGMENT.
*                CNI_E_BAD_PACKET - The Packet is not a CNIPACKET.
*                CNI_E_FRAGMENT_TOO_SHORT - The ulLength is greater than
*                the maximum length of this fragment.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXSetFragmentLength(IN OUT CNIFRAGMENT Fragment,
                           IN ULONG ulLength, IN OUT CNIPACKET Packet)
{
    UINT ulOriginalFragmentSize = 0;
    UINT ulInitializeSize;
    LPFRAGMENTBUFFER lpFragmentBuffer = (LPFRAGMENTBUFFER) Fragment;
    LPPACKETDESCRIPTOR lpPacketDescriptor;

    if (!lpFragmentBuffer)
        return CNI_E_BAD_FRAGMENT;

    QueryBuffer(lpFragmentBuffer, NULL, &ulInitializeSize);

    if (ulInitializeSize < ulLength)
        return CNI_E_FRAGMENT_TOO_SHORT;

    ulOriginalFragmentSize = lpFragmentBuffer->uiFragmentDataSize;
    lpFragmentBuffer->uiFragmentDataSize = ulLength;

    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;

    if (lpPacketDescriptor) {
        /* adjust packet descripter information */
        lpPacketDescriptor->uiPacketSize -= ulOriginalFragmentSize;
        lpPacketDescriptor->uiPacketSize += ulLength;
    }

    return CNI_SUCCESS;
}



/*************************************************************************
* 
* CNI_LINUXGetFragmentInfo
* 
* Description:
* This function will retrieve the data pointer for a fragment, and the length
* of that data.
* 
* 
* Argument:	
* 
*   IN CNIFRAGMENT Fragment - A packet fragment.
*   OUT PCHAR *ppData - A pointer to a character pointer that 
*                will contain the fragment data pointer.
*   OUT ULONG *pulLength - A pointer to the length of the fragment.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The Fragment length was changed.
*                CNI_E_BAD_FRAGMENT - The Fragment is not a CNIFRAGMENT.
*                CNI_E_BAD_PARAMETER - One of the parameters was invalid,
*                i.e., NULL.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXGetFragmentInfo(IN CNIFRAGMENT Fragment,
                         OUT PCHAR * ppData, OUT ULONG * pulLength)
{

    LPFRAGMENTBUFFER lpFragmentBuffer = (LPFRAGMENTBUFFER) Fragment;

    *ppData = lpFragmentBuffer->lpFragmentData;

    *pulLength = lpFragmentBuffer->uiFragmentDataSize;

    return CNI_SUCCESS;
}

  /*************************************************************************
*
* CNI_LINUXGetFrameType
*
* Description:
* This function retrieves the media type for the packets sent and received
* via this medium.
*
*	
* Argument:
*
*   IN CNIBINDING Binding - A binding handle.
*   OUT PNICMEDIUM pMedium - The media type for this binding.
*                CniMedium802_3	Ethernet
*                CniMedium802_5	Token Ring
*                CniMediumFddi	Fiber Distributed Data Interface
*                CniMediumWan	Various point to point and WAN interfaces
*                CniMediumLocalTalk	LoacalTalk network (MAC)
*                CniMediumDix	Ethernet - drivers use the DIX header format.
*                CniMediumArcnetRaw	ARCNET network
*                CniMediumArcnet878_2	ARCNET 878.2 network
*                CniMediumAtm	ATM network
*                CniMediumWirelessWan	Various wireless WAN networks
*                CniMediumIrda	IRDA, infrared network
*
*  NOTE - Only CniMediumUNKNOWN, CniMedium802_3 and CniMediumWan will be
*         returned at present as only the second 2 are accepted. Other
*         is for errors (shouldn't happen).
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - The Frame type was returned.
*                CNI_E_BAD_BINDING - The Binding is not a CNIBINDING.
*                CNI_E_BAD_PARAMETER - pMedium was NULL.
*
*************************************************************************/
CNISTATUS
CNI_LINUXGetFrameType(IN CNIBINDING Binding, OUT PNICMEDIUM pMedium)
{

    // We will support 802_3 for now.
    *pMedium = CniMedium802_3;


    return CNI_SUCCESS;
}

 /*************************************************************************
*
* CNI_LINUXGetMacAddress
*
* Description:
* This function retrieves the MAC address for the binding specified.
* The character array represented by *ppMacAddress must not be modified,
* or released.
*
*
* Argument:	
*
*   IN CNIBINDING Binding - A binding handle
*   OUT PCHAR *ppMacAddress - A pointer to a character array that
*                contains the MAC address.
*   OUT ULONG pulMacAddressSize - The Fragment length was changed.
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - Returned the MAC address.
*                CNI_E_BAD_BINDING - The Binding is not a CNIBINDING.
*                CNI_E_BAD_PARAMETER - pMedium was NULL.
*
*************************************************************************/
CNISTATUS
CNI_LINUXGetMacAddress(IN CNIBINDING Binding,
                       OUT PCHAR * ppMacAddress,
                       OUT ULONG * pulMacAddressSize)
{
    //  this code will be different in 2.4.x kernel
    PBINDING pBinding;

    if (!Binding)
        return CNI_E_BAD_BINDING;

    pBinding = (PBINDING) Binding;

    if (!ppMacAddress || !pulMacAddressSize)
        return CNI_E_BAD_PARAMETER;


    *ppMacAddress = pBinding->pDevice->dev_addr;

    if(ippp_dev(pBinding->pDevice)||(pBinding->pDevice->hard_header_len == 4)){
        *pulMacAddressSize = ETH_ALEN;
    }
    else if (pBinding->pDevice->hard_header_len == ETH_HLEN) {
        *pulMacAddressSize = pBinding->pDevice->addr_len;
    }

    return CNI_SUCCESS;
}

/*************************************************************************
*
* CNI_LINUXGetMacName
*
* Description:
* This function retrieves the name of the network device associated with a
* given binding.  It works differently depending on the operating system.
*
* On NT systems, the MAC name will be the full device name of the NIC card,
* as seen by the IPCONFIG.EXE application.  The name string on NT includes
* the full device driver name for the NIC driver, which has the form:
* "DosDevices\<driver name>".  The registry settings for the device can be
*  found in the registry at:
* "HKEY_LOCAL_MACINE\System\CurrentControlSet\Services\<driver name>"
*
* On Windows 95/98 systems, the name string is an enumerated string
* corresponding to the NIC driver.  Settings for the NIC driver are found in
* the registry at:
* "HKEY_LOCAL_MACHINE\System\CurrentControlSet\Services\Class\Net\<enumerated string>"
*
*
*
* Argument:
*	
*   IN CNIBINDING Binding - A binding handle that will be used to determine
*                           the associated network adapter
*   OUT PCHAR *macName - a pointer to the character string giving the name
*                        of the name associated with Binding
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - The packet was sent by the NIC.
*                CNI_E_BAD_PARAMETER - One of the parameters was invalid
*
*************************************************************************/
CNISTATUS
CNI_LINUXGetMacName(IN CNIBINDING Binding, OUT PCHAR * pszMacName)
 {
    PBINDING pBinding;

    if (!Binding || !pszMacName)
        return CNI_E_BAD_PARAMETER;

    pBinding = (PBINDING) Binding;

    *pszMacName = pBinding->pDevice->name;

    return CNI_SUCCESS;
}

/*************************************************************************
*
* CNI_LINUXGetMTUSize
*
* Description:
* This function retrieves the MTU for the specified binding.
*
*	
* Argument:
*
*   IN CNIBINDING Binding - A binding handle
*   OUT PULONG pulMtuSize - The size, in bytes, of the MTU for
*                this binding.
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - The MTU was retrieved.
*                CNI_E_BAD_BINDING - The Binding is not a CNIBINDING.
*                CNI_E_BAD_PARAMETER - pulMtuSize was NULL.
*
*************************************************************************/
CNISTATUS CNI_LINUXGetMTUSize(IN CNIBINDING Binding, OUT PULONG pulMtuSize)
{
    PBINDING pBinding;

    if (!Binding || !pulMtuSize)
        return CNI_E_BAD_PARAMETER;

    pBinding = (PBINDING) Binding;

    /* return the original MTU */
    *pulMtuSize = (ULONG) pBinding->original_mtu;

    return CNI_SUCCESS;
}






/*************************************************************************
* 
* CNI_LINUXInjectReceive
* 
* Description:
* This function will inject a Packet into the receive data stream as though
* they were received by and send to a NIC/protocol pair defined by Binding.
* The ReceiveContext should be set to provide packet-specific information 
* to CniTransferData() and CniReceiveComplete().  This plugin will not see
* the injected packet at its CniReceive() interface, it will be sent 
* directly to the protocol.
* If the size of Packet is less than ulSize, and the receiving protocol is
* interested in receiving the rest of the packet the plugin's
* CniTransferData() entry point will be called to copy the remaining packet
* data into a Packet belonging to the protocol.
* 
* Argument:
* 
*   IN CNIBINDING Binding - A binding handle that will be used to
*                receive this packet.
*   IN PVOID ReceiveContext -	Data to be passed to the 
*                TransferData entry point.
*   IN CNIFRAGMENT MacHeader - This is the header that will be 
*                copied to the packet before it is received by the protocol.
*                It should correspond to the Binding.
*   IN CNIPACKET Packet - The packet that will be received, minus 
*                the MAC Header.
*   IN ULONG ulSize - The size of the complete packet.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The packet was sent
*                CNI_PENDING - The packet was not sent yet, but will be.
*                CNI_W_BAD_FORMAT - The packet data was formatted incorrectly.
*                CNI_E_NO_ENTRY_POINT - The plugin did not have a 
*                CniTransferData() entry point.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXInjectReceive(IN CNIBINDING Binding,
                       IN PVOID ReceiveContext,
                       IN CNIFRAGMENT MacHeader,
                       IN CNIPACKET Packet, IN ULONG ulSize)
{
    CNISTATUS rc = CNI_SUCCESS;
    LPPACKETDESCRIPTOR lpPacketDescriptor;
    PBINDING pBinding;
    LPFRAGMENTBUFFER lpMacFragment;
    struct sk_buff *skb = NULL;
    unsigned char *pIP = NULL, *pMac = NULL;


    /* we need to build the actual sk_buff from the packet structure */
    pBinding = (PBINDING) Binding;
    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;
    lpMacFragment = (LPFRAGMENTBUFFER) MacHeader;

    skb = dev_alloc_skb(lpPacketDescriptor->uiPacketSize
                        + lpMacFragment->uiFragmentDataSize);
    if (!skb) {
        /* need to call  ReceiveComplete otherwise ReceiveContext will not be free) */
        rc = CNI_W_OUT_OF_DESCRIPTORS;
    } else {
        /* move the data into the packet */
        do_gettimeofday(&skb->stamp);

        pIP = skb_put(skb, lpPacketDescriptor->uiPacketSize);

        CniGetPacketData(Packet, 0, lpPacketDescriptor->uiPacketSize, pIP);

        skb->dev = pBinding->pDevice;

        /* put back the mac header */
        if (pBinding->pDevice->hard_header_len == ETH_HLEN) {
            pMac = skb_push(skb, lpMacFragment->uiFragmentDataSize);
            memcpy(pMac, lpMacFragment->lpFragmentData,
                   lpMacFragment->uiFragmentDataSize);

            skb->protocol = eth_type_trans(skb, skb->dev);

        } else if(ippp_dev(pBinding->pDevice)||pBinding->pDevice->hard_header_len==4) {
            pMac = pIP;

            skb->protocol = htons(ETH_P_IP);
        }
        skb->dev = pBinding->pDevice;



        skb->ip_summed = CHECKSUM_UNNECESSARY;

        skb->nh.iph = (struct iphdr *) skb->data;
        skb->mac.raw = pMac;

        if (pBinding->dst) {
            /* Only loop back device will have pBinding->dst set.
               This packet must be a IPC UDP packet. 
               All loop back packet must have dst set otherwise the packet will
               be droped by the IP layer. */
            skb->dst = dst_clone(pBinding->dst);
        } else {
            pBinding->recv_stat.called = TRUE;
        }

        pBinding->recv_stat.rc =
            pBinding->InjectReceive(skb, skb->dev, pBinding->pPT);
    }

    CNICallbackTable.ReceiveComplete(Binding, ReceiveContext, Packet);
    return rc;
}


/*************************************************************************
* 
* CNI_LINUXInjectSend
* 
* Description:
* This call will inject a packet into the send data stream, the packet will
* appear to be originating from and sent via a protocol/NIC pair defined by
* Binding.  This plugin will not see the packet in it CniSend() entry point,
* but will receive notification that the packet has been sent via it's 
* CniSendComplete() entry point.  The SendContext will be used to send 
* packet-specific data to CniSendComplete so that this packet may be 
* identified as belonging to this plugin.
*
* 
* 
* Argument:
*	
*   IN DNBINDING Binding - A binding handle that will be used
*                 to receive this packet.
*   IN PVOID SendContext - Data to be passed to the SendComplete
*                 entry point.
*   IN DNFRAGMENT MacHeader - This is the header that will be 
*                copied to the packet before it is sent over the NIC.  It 
*                should correspond to the Binding.
*   IN DNPACKET Packet - The packet that will be sent, minus the
*                 MAC header.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The packet was sent by the NIC.
*                CNI_PENDING - The packet will be sent by theNIC.
*                CNI_W_BAD_FORMAT - The packet data was not formatted 
*                correctly for the NIC.
* 
*************************************************************************/
CNISTATUS
CNI_LINUXInjectSend(IN CNIBINDING Binding,
                    IN PVOID SendContext,
                    IN CNIFRAGMENT MacHeader, IN CNIPACKET Packet)
{
    CNISTATUS rc = CNI_SUCCESS;
    LPPACKETDESCRIPTOR lpPacketDescriptor;
    PBINDING pBinding;
    LPFRAGMENTBUFFER lpMacFragment;
    struct sk_buff *skb;
    unsigned char *pIP=NULL, *pMac=NULL;

    /* we need to build the actual sk_buff from the packet structure */
    pBinding = (PBINDING) Binding;
    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;
    lpMacFragment = (LPFRAGMENTBUFFER) MacHeader;

    /*XXX somebody write a comment about the + 2 on this call... */
    skb = dev_alloc_skb(lpPacketDescriptor->uiPacketSize
                        + lpMacFragment->uiFragmentDataSize + 2);

    if (!skb) {
        rc = CNI_W_OUT_OF_DESCRIPTORS;
    } else {
        /* transfer the packet data into sk_buff */
        
        if(ippp_dev(pBinding->pDevice)) {
            pMac=skb_put(skb,IPPP_MAX_HEADER);
        }
        else if(pBinding->pDevice->hard_header_len == ETH_HLEN) {
            pMac = skb_put(skb, lpMacFragment->uiFragmentDataSize); 
            memcpy(pMac, lpMacFragment->lpFragmentData,
                   lpMacFragment->uiFragmentDataSize);
        }
        else if(pBinding->pDevice->hard_header_len == 4) {
            pMac = skb->data;
        }

        pIP = skb_put(skb, lpPacketDescriptor->uiPacketSize);
        CniGetPacketData(Packet, 0, lpPacketDescriptor->uiPacketSize, pIP);

        /* put the mac header on */
        do_gettimeofday(&skb->stamp);

        skb->dev = pBinding->pDevice;

        skb->mac.raw = pMac;
        skb->nh.raw = pIP;

        /*ip header length is in 32bit words */
        skb->h.raw = pIP + (skb->nh.iph->ihl * 4);
        skb->protocol = htons(ETH_P_IP);

        /* send this packet up the NIC driver */
        pBinding->send_stat.rc = pBinding->InjectSend(skb, skb->dev);
        /* if the nic's hard_start_xmit function failed,
           the kernel will queue original packet
           and send to us again.  So, we free the packet that was just built,
           and when the kernel calls dev->hard_start_xmit, we'll start all
           over again... see sch_generic.c:qdisc_restart() for details.
        */
        if (pBinding->send_stat.rc != 0)
        {
            dev_kfree_skb(skb);
        }
        pBinding->send_stat.called = TRUE;
    }

    /* we have to call Sendcomplete here or else the sendcontext will
       * not be free */
    CNICallbackTable.SendComplete(pBinding, SendContext, Packet);
    return rc;
}


CNIBINDING CNI_LINUXGetTrueBinding(IN CNIBINDING Binding)
{
    PBINDING pBinding = (PBINDING) Binding;
    return (CNIBINDING) pBinding->pDevice;
}

CNIBINDING CNI_LINUXGetBindingforIpcUdp(IN CNIBINDING Binding)
{
    return (CNIBINDING) &BindingIpcUdp;
}
