unit FFT; { 1997 07 13/15; 1998 01 14 }

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, ExtCtrls;

const
  MaxFFTSize=2048;
  PARZEN=1;
  SQUARE=2;
  WELCH=3;
  AbsLevel=0.095;  { Poziom do pseudologarytmicznej skali }

type
  FFTTABLE=array[1..MaxFFTSize] of single;
  TWOFFT=array[1..2*MaxFFTSize] of single;
  TABLE=array[1..512] of integer;

  TFormFFT = class(TForm)
    WindowRadioGroup: TRadioGroup;
    CloseButton: TBitBtn;
    Image1: TImage;
    FreqLabel: TEdit;
    BitBtn1: TBitBtn;
    BitBtn2: TBitBtn;
    LogBox: TCheckBox;
    GridBox: TCheckBox;
    ZoomLabel: TLabel;
    procedure CloseButtonClick(Sender: TObject);
    procedure FFTMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FFTPaintForm(Sender: TObject);
    procedure FormFFTCreate(Sender: TObject);
    procedure WindowRadioGroupClick(Sender: TObject);
    procedure BitBtn1Click(Sender: TObject);
    procedure BitBtn2Click(Sender: TObject);
    procedure LogBoxClick(Sender: TObject);
    procedure GridBoxClick(Sender: TObject);
    procedure FormFFTDestroy(Sender: TObject);
    procedure FormFFTOnKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);

  private
                                          { Private declarations }
     SamplingFreq          : single;
     LoadStat              : boolean;
     dataFFT,OrgDataFFT    : FFTTABLE;
     lenFFT                : integer;
     typFFT                : integer;
     ZoomFactor            : integer;
     LogStat,GridStat      : boolean;
     xa,y2                 : FFTTABLE;
     ScaleY,Level          : single;
     CanMove               : boolean;
     Value                 : TABLE;

   procedure Computing;
   procedure Rescale;
   procedure SetValueTable;

  public
    { Public declarations }
    procedure SetFFT(var signal : FFTTABLE; n : integer;
                          sampling : single);
  end;

PROCEDURE four1(VAR data: TWOFFT; nn,isign: integer);
FUNCTION window(j: integer; facm,facp: single; typ : integer): single;
function SetLenFFT(n : integer) : integer;

var
  FormFFT: TFormFFT;

implementation

uses
  FIRST;
               { Funkcje z recept numerycznych }

FUNCTION sngl(x:single):single;
  BEGIN
    sngl := x
  END;
             { Transformata Fouriera (FFT) }
PROCEDURE four1(VAR data: TWOFFT; nn,isign: integer);
VAR
   ii,jj,n,mmax,m,j,istep,i: integer;
   wtemp,wr,wpr,wpi,wi,theta: double;
   tempr,tempi: single;
BEGIN
   n := 2*nn;
   j := 1;
   FOR ii := 1 TO nn DO BEGIN
      i := 2*ii-1;
      IF (j > i) THEN BEGIN
         tempr := data[j];
         tempi := data[j+1];
         data[j] := data[i];
         data[j+1] := data[i+1];
         data[i] := tempr;
         data[i+1] := tempi
      END;
      m := n DIV 2;
      WHILE ((m >= 2) AND (j > m))  DO BEGIN
         j := j-m;
         m := m DIV 2
      END;
      j := j+m
   END;
   mmax := 2;
   WHILE (n > mmax) DO BEGIN
      istep := 2*mmax;
      theta := 6.28318530717959/(isign*mmax);
      wpr := -2.0*sqr(sin(0.5*theta));
      wpi := sin(theta);
      wr := 1.0;
      wi := 0.0;
      FOR ii := 1 TO (mmax DIV 2) DO BEGIN
         m := 2*ii-1;
         FOR jj := 0 TO ((n-m) DIV istep) DO BEGIN
            i := m + jj*istep;
            j := i+mmax;
            tempr := sngl(wr)*data[j]-sngl(wi)*data[j+1];
            tempi := sngl(wr)*data[j+1]+sngl(wi)*data[j];
            data[j] := data[i]-tempr;
            data[j+1] := data[i+1]-tempi;
            data[i] := data[i]+tempr;
            data[i+1] := data[i+1]+tempi
         END;
         wtemp := wr;
         wr := wr*wpr-wi*wpi+wr;
         wi := wi*wpr+wtemp*wpi+wi
      END;
      mmax := istep
   END
END;
            { Okienkowane }
FUNCTION window(j: integer; facm,facp: single; typ : integer): single;
  var
    tmp : single;

BEGIN
 case typ of
   PARZEN: tmp:=(1.0-abs(((j-1)-facm)*facp));         (* Parzen *)
   SQUARE: tmp:=1.0;                                  (* Square *)
   WELCH:  tmp:=(1.0-sqr(((j-1)-facm)*facp));         (* Welch  *)
 end;
 window:=tmp;
END;
          { Spaljny kubiczne }
PROCEDURE spline(VAR x,y: FFTTABLE; n: integer; yp1,ypn: single;
                 VAR  y2: FFTTABLE);
VAR
   i,k: integer;
   p,qn,sig,un: single;
   u: ^FFTTABLE;

BEGIN
   MemInit(u,sizeOf(FFTTABLE));
   IF (yp1 > 0.99e30) THEN BEGIN
      y2[1] := 0.0;
      u^[1] := 0.0
   END ELSE BEGIN
      y2[1] := -0.5;
      u^[1] := (3.0/(x[2]-x[1]))*((y[2]-y[1])/(x[2]-x[1])-yp1)
   END;
   FOR i := 2 TO n-1 DO BEGIN
      sig := (x[i]-x[i-1])/(x[i+1]-x[i-1]);
      p := sig*y2[i-1]+2.0;
      y2[i] := (sig-1.0)/p;
      u^[i] := (y[i+1]-y[i])/(x[i+1]-x[i])
         -(y[i]-y[i-1])/(x[i]-x[i-1]);
      u^[i] := (6.0*u^[i]/(x[i+1]-x[i-1])-sig*u^[i-1])/p
   END;
   IF (ypn > 0.99e30) THEN BEGIN
      qn := 0.0;
      un := 0.0
   END ELSE BEGIN
      qn := 0.5;
      un := (3.0/(x[n]-x[n-1]))*(ypn-(y[n]-y[n-1])/(x[n]-x[n-1]))
   END;
   y2[n] := (un-qn*u^[n-1])/(qn*y2[n-1]+1.0);
   FOR k := n-1 DOWNTO 1 DO
    BEGIN
      y2[k] := y2[k]*y2[k+1]+u^[k]
    END;
   MemFree(u,sizeOf(FFTTABLE));
END;

PROCEDURE splint(VAR xa,ya,y2a: FFTTABLE; n: integer;
                 x: single; VAR y: single);
VAR
   klo,khi,k: integer;
   h,b,a: single;

BEGIN
   klo := 1;
   khi := n;
   WHILE (khi-klo > 1) DO BEGIN
      k := (khi+klo) DIV 2;
      IF (xa[k] > x) THEN khi := k ELSE klo := k
   END;
   h := xa[khi]-xa[klo];
   a := (xa[khi]-x)/h;
   b := (x-xa[klo])/h;
   y := a*ya[klo]+b*ya[khi]+
      ((a*a*a-a)*y2a[klo]+(b*b*b-b)*y2a[khi])*(h*h)/6.0
END;
                                 { Obliczenia }
procedure TFormFFT.Computing;
  var
    facm,facp        : single;
    i                : integer;
    tmptab           : ^TWOFFT;
    B                : boolean;
    Pmin,Pmax,scale  : single;

begin
 B:=MemInit(tmptab,SizeOf(TWOFFT));
 if not B then
   begin
     LoadStat:=false;
     exit;
   end;

 facm:=lenFFT-0.5;
 facp:=1.0/(lenFFT+0.5);

 for i:=1 to lenFFT do     { Przygotawanie danych dla FFT }
  begin
   tmptab^[2*i-1]:=OrgDataFFT[i]*window(2*i-1,facm,facp,typFFT);
   tmptab^[2*i]:=0.0;
  end;

 four1(tmptab^,lenFFT,1);  { FFT }
 for i:=1 to lenFFT do     { Wyznaczenie modulu widma }
   DataFFT[i]:=sqrt(sqr(tmptab^[2*i-1])+sqr(tmptab^[2*i]));

 Pmin:=DataFFT[1];
 Pmax:=Pmin;
 for i:=1 to lenFFT do
   begin
     if Pmin>DataFFT[i] then Pmin:=DataFFT[i];
     if Pmax<DataFFT[i] then Pmax:=DataFFT[i];
   end;

 if Pmax<>Pmin then
   scale:=1.0/(Pmax-Pmin)
 else
   scale:=1.0;

 for i:=1 to lenFFT do    { Przeskalaowanie danych na 0..1 }
  begin
    DataFFT[i]:=scale*(DataFFT[i]-Pmin);
    if LogStat then
      DataFFT[i]:=ln(AbsLevel+DataFFT[i]);
   end;

 MemFree(tmptab,sizeOf(TWOFFT));
 for i:=1 to lenFFT do     { Generowanie splajnow kubicznych }
   xa[i]:=i-1;
 spline(xa,dataFFT,lenFFT,dataFFT[2]-dataFFT[1],
        dataFFT[lenFFT div 2]-dataFFT[lenFFT div 2-1] ,y2);
 Rescale;                  { Ustalenie skal  }
end;

{$R *.DFM}

function SetLenFFT(n : integer) : integer;
  var
    tmp : integer;    { Dlugosc odcinka dla FFT }

begin
  if (n>=0) and (n<=256) then
    tmp:=256
  else if (n>256) and (n<=512) then
    tmp:=512
  else if (n>512) and (n<=1024) then
    tmp:=1024
  else
    tmp:=2048;
  SetLenFFT:=tmp;
end;

procedure TFormFFT.SetFFT(var signal : FFTTABLE; n : integer;
                          sampling : single);
var
  i,k  : integer;
  sum  : single;

begin
 LoadStat:=true;
 SamplingFreq:=0.5*sampling;
 lenFFT:=SetLenFFT(n);     { Ladowanie danych }
 for i:=1 to n do
   if i<=MaxFFTSize then
     OrgDataFFT[i]:=signal[i];

 sum:=0.0;
 k:=1;
 for i:=1 to n do          { Skladwa stala }
   if i<=MaxFFTSize then
    begin
     sum:=sum+OrgDataFFT[i];
     k:=k+1;
    end;
 sum:=sum/k;

 for i:=1 to lenFFT do     { Eliminacja skladowaj stalej + inwersja ? }
  if i<=n then
    OrgDataFFT[i]:=sum-OrgDataFFT[i]
  else
    OrgDataFFT[i]:=0.0;    { Poza obszarem sygnalu 0.0 }

 Computing;
 SetValueTable;
 Repaint;
end;

procedure TFormFFT.CloseButtonClick(Sender: TObject);
begin
   LoadStat:=false;       { Opuszczenie okienka }
   Close;
end;

procedure TFormFFT.FFTMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
  var                     { Pozycja kursora na widmie }
    nap            : string;

begin
  if not LoadStat then
    exit;
  if not CanMove then
    exit;

  Str(SamplingFreq*X/(Image1.Width*ZoomFactor):8:2,nap);
  FreqLabel.Text:=nap+' [Hz]';
end;

procedure TFormFFT.Rescale;
  var
    i                 : integer;
    Pmin,Pmax,scale,y : single;

begin                      { Skalowanie }
   scale:=((lenFFT div 2) div ZoomFactor)/Image1.Width;
   splint(xa,dataFFT,y2,lenFFT,0.0,y);
   Pmax:=y;
   for i:=1 to Image1.Width do
     begin
       splint(xa,dataFFT,y2,lenFFT,scale*(i-1),y);
       if Pmax<y then Pmax:=y;
     end;

   if LogStat then               { Na przedziale 0..Pmax -> 0..1 }
     Pmin:=ln(AbsLevel)
   else
     Pmin:=0.0;

   if Pmin<>Pmax then
     begin                       { Odwzorowanie liniowe }
      ScaleY:=1.0/(Pmax-Pmin);
      Level:=-ScaleY*Pmin;
     end
   else
     begin
      ScaleY:=1.0;
      Level:=0.0;
     end;
end;

procedure TFormFFT.SetValueTable;
  var
    i       : integer;   { Ustalenie wartosci widma na ekranie }
    y,scale : single;

begin
   rescale;
   scale:=((lenFFT div 2) div ZoomFactor)/Image1.Width;
   for i:=1 to Image1.Width do
    begin
     splint(xa,dataFFT,y2,lenFFT,scale*(i-1),y);
     Value[i]:=round(Image1.Height-Image1.Height*(ScaleY*y+Level));
    end;
end;

procedure TFormFFT.FFTPaintForm(Sender: TObject);
  type
     VALTAB=array[0..10] of single;

  var
    i,skok             : integer;     { Rysowanie widma }
    scale,fjump,y      : single;
    itmp               : integer;
    OldColor,OldBColor : TColor;
    OldFont            : TFont;
    napis              : string;
    Pmin,Pmax          : single;
    tmpTab             : VALTAB;

begin
   if not LoadStat then
     exit;
   CanMove:=false;
   fjump:=10.0;
   with Image1 do
     begin
       OldColor:=brush.Color;
       brush.Color:=clWhite;        { Wypelnianie prostokata }
       Canvas.rectangle(0,0,Width,Height);
       brush.Color:=OldColor;

       OldColor:=Canvas.pen.Color;  { Ramka obrazka }
       Canvas.pen.Color:=clBlack;
       Canvas.rectangle(0,0,Width,Height);
       Canvas.pen.Color:=OldColor;

       OldFont:=FormFFT.Font;

       with FormFFT.Font do
         begin
           Color:=clBlue;
           size:=7;
           Name:='Times New Roman CE';
           Style:=[fsBold];
         end;

       case ZoomFactor of          { Zmiana opisu skali czestotliwosci }
         1:    fjump:=10.0;
         2..4: fjump:=5.0;
         5..8: fjump:=2.5;
         9..16: fjump:=1.0;
        else
         fjump:=0.1;
       end;

       skok:=round(ZoomFactor*fjump*Width/SamplingFreq);
       itmp:=Width div skok;
       OldBColor:=FormFFT.Canvas.brush.Color;
       FormFFT.Canvas.brush.Color:=brush.Color;
       for i:=0 to itmp do            { Rzmki + opisy }
         begin
           Canvas.moveTo(skok*i,Height);
           Canvas.lineTo(skok*i,0);
           Str(fjump*i:4:2,napis);    { Pozioma }
           FormFFT.Canvas.TextOut(40+skok*i-FormFFT.Font.size*
                                  (length(napis) div 2),Height+25, { * }
                                  napis);
         end;

       if gridstat then              { Siatka }
         begin
           OldColor:=Canvas.pen.Color;
           Canvas.pen.Color:=clGray;
           itmp:=Width div skok;
           for i:=1 to itmp do       { Pionowe kreski }
             begin
               Canvas.moveTo(skok*i,Height);
               Canvas.lineTo(skok*i,3);
             end;
         end;

         if not LogStat then
            for i:=1 to 10 do
              begin
               itmp:=Height-round(0.10*Height*i);
               if GridStat then
                 begin               { Siateczka }
                   Canvas.moveTo(0,itmp);
                   Canvas.lineTo(Width,itmp);
                  end
               else
                begin
                  Canvas.moveTo(0,itmp);
                  Canvas.lineTo(3,itmp);
                end;

                Str(10*i:3,napis);    { Opis osi }
                if i<>0 then
                  FormFFT.Canvas.TextOut(15,itmp+15,napis); { * } 
              end
            else                     { Skala pseudo-logarytmiczna }
              begin
                for i:=0 to 10 do
                  tmpTab[i]:=ln(AbsLevel+0.1*i);

                Pmin:=tmpTab[0];     { Rysowanie skali logarytmicznej }
                Pmax:=tmpTab[10];
                scale:=Height/(Pmax-Pmin);
                for i:=1 to 10 do
                 begin
                  itmp:=Height-round(scale*(tmpTab[i]-Pmin));
                  if GridStat then                 { Siateczka }
                    begin
                     Canvas.moveTo(0,itmp);
                     Canvas.lineTo(Width,itmp);
                    end
                  else
                    begin
                      Canvas.moveTo(0,itmp);
                      Canvas.lineTo(3,itmp);
                    end;

                  Str(10.0*i:3:0,napis);          { Opis osi }
                  if i<>0 then
                    FormFFT.Canvas.TextOut(15,itmp+10,napis);
                 end;
              end;
              Canvas.pen.Color:=OldColor;

       FormFFT.Canvas.brush.Color:=OldBColor;
       FormFFT.Font:=OldFont;

       OldColor:=Canvas.pen.Color;
       Canvas.pen.Color:=clRed;  { Rysowanie widma }
       Canvas.moveTo(1,Value[1]);
       for i:=1 to Width do
         Canvas.lineTo(i,Value[i]);
       Canvas.pen.Color:=OldColor;
     end;

   if ZoomFactor<>1 then
     ZoomLabel.Caption:='Zoom '+IntToStr(ZoomFactor)+'x'
   else
     ZoomLabel.Caption:='';
   CanMove:=true;
end;

procedure TFormFFT.FormFFTCreate(Sender: TObject);
begin
  LoadStat:=false;           { Inicjacja }
  typFFT:=SQUARE;
  WindowRadioGroup.ItemIndex:=0;
  ZoomFactor:=1;
  LogStat:=false;
  GridStat:=true;
  CanMove:=false;

  FreqLabel.Font.Color:=clBlack;
  FreqLabel.Text:='  0.00 [Hz]';

  Image1.Cursor:=crCross;
end;

procedure TFormFFT.WindowRadioGroupClick(Sender: TObject);
  var
    name : string;   { Okienkowanie }

begin
  name:=WindowRadioGroup.Items.Strings[WindowRadioGroup.ItemIndex];
  if name='  Parzen' then
    typFFT:=PARZEN               { UWAGA na te napisy !!! }
  else if name='  Square' then
    typFFT:=SQUARE
  else if name='  Welch' then
    typFFT:=WELCH
  else typFFT:=SQUARE;

  if LoadStat then
    begin
      Computing;
      SetValueTable;
      Repaint;
    end;
end;

procedure TFormFFT.BitBtn1Click(Sender: TObject);
 var
   OldFactor : integer;  { Zoom - }

begin
  OldFactor:=ZoomFactor;
  Zoomfactor:=ZoomFactor div 2;
  if Zoomfactor<1 then
    ZoomFactor:=1;
  if OldFactor<>ZoomFactor then
   begin
     SetValueTable;
     Repaint;
   end;
end;

procedure TFormFFT.BitBtn2Click(Sender: TObject);
  var
    OldFactor : integer;   { Zoom + }

begin
  OldFactor:=ZoomFactor;
  ZoomFactor:=ZoomFactor*2;
  if ZoomFactor>32 then
    ZoomFactor:=32;
  if OldFactor<>ZoomFactor then
   begin
    SetValueTable;
    Repaint;
   end;
end;

procedure TFormFFT.LogBoxClick(Sender: TObject);
begin
  if LogBox.Checked then { Skala logarytmiczna }
    LogStat:=true
  else
    LogStat:=false;

  Computing;
  SetValueTable;
  Repaint;
end;

procedure TFormFFT.GridBoxClick(Sender: TObject);
begin
  if GridBox.Checked then   { Siateczka }
    GridStat:=true
  else
    GridStat:=false;
  Repaint;
end;

procedure TFormFFT.FormFFTDestroy(Sender: TObject);
begin
  LoadStat:=false;        { Opuszczanie okienka }
  CanMove:=false;
end;

procedure TFormFFT.FormFFTOnKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key=VK_ESCAPE then
    CloseButtonClick(Sender);
end;

end.
