; MltBoot - Multiple Boots in single ROM (many bootroms packed)
;
; define TESTMODE to produce test version
;
; idea about packing - records consisting of word count, then
;   odd records: count bytes, no more than 5 consecutive identical;
;   even records: no data, last byte from previous record is to be
;		  repeated count times;
;   a count 0 means end of the bootrom image
; every packed bootrom image should be preceded by
;   a word specifying offset to next bootrom (first free cell in
;     last, free cells are 'FF'-s so they can be reprogrammed)
;   a zero-terminated description to be displayed in menu
; and may be followed by any data before location where the next
; bootrom image starts (it is ignored), usually by checksum byte
;
; menu: F-boot from floppy, H-boot from HD, 1..9-use bootrom 1..9
; default (on timeout):
;
; Definitions:
Debugging	=	0		; non-zero displays debug marks
	ifdef	TESTMODE
BootLoadOffset	=	4000h
BootLoadSegment	=	7000h
RomLoadSegment	=	7000h
TestLoadSegment	=	6000h
ComLoadOffset	=	100h
	else
BootLoadOffset	=	7C00h		; load boot sector to 0:7C00h
BootLoadSegment	=	0
RomLoadSegment	=	2000h
	endif
BootSign_1	=	55h
BootSign_2	=	0aah
SectorSize	=	200h
PageSize	=	200h
int_biosvideo	=	10h
biosvideo_wrtty	=	0eh
biosvideo_yellow=	0eh
int_biosmem	=	12h
int_biosdisk	=	13h
disk_timeout	=	80h
hard_disk	=	80h
diskf_query	=	8
int_bioskey	=	16h
int_biostime	=	1ah
int_reboot	=	19h
int_bootsav	=	0c0h
bootsav_sign	=	424dh		; "Boot Menu" or "Multi Boot"
ch_cr		=	0dh
ch_lf		=	0ah
biosdisk_read1	=	0201h

; Macros:
puss		macro	r1,r2,r3,r4,r5,r6,r7,r8,r9,ra,rb,rc	; push many
		irp	rx,<rc,rb,ra,r9,r8,r7,r6,r5,r4,r3,r2,r1>
		ifnb	<rx>
		push	rx
		endif
		endm
		endm
popp		macro	r1,r2,r3,r4,r5,r6,r7,r8,r9,ra,rb,rc	; pop many
		irp	rx,<r1,r2,r3,r4,r5,r6,r7,r8,r9,ra,rb,rc>
		ifnb	<rx>
		pop	rx
		endif
		endm
		endm
debug		macro	char
		ifdef	DebugProc
		call	DebugProc
		db	'&char&'
		endif
		endm

Text		segment	para public 'code'
		assume	cs:Text

RomHeader	label	byte
	ifdef	TESTMODE
		jmp	short RomBootInit_n
	else
		db	BootSign_1,BootSign_2
	endif
RomPageCount	db	0		; RomPages to be filled later

RomBootInit	proc	far		; sets INT 19h vector if not set yet
		jmp	short RomBootInit_n
SelectTimeout	db	0
NextRomOffset	dw	EndInitModule
RomBootInit_n	label	near		; all registers preserved
	ifdef	TESTMODE
		mov	si,ComLoadOffset
		mov	ax,TestLoadSegment
		mov	es,ax
		xor	di,di
		mov	cx,-ComLoadOffset
		cld
		rep	movsb
		mov	ax,offset RomBootVect
		push	es
		push	ax
		ret
	else
		puss	es,bx,ax
		mov	al,int_bootsav+1
		call	IntVectAddr
		mov	ax,bootsav_sign
		cmp	es:[bx],ax
		jz	RomBootInit_x
		mov	es:[bx],ax
		mov	ax,(int_bootsav shl 8)+int_reboot
		call	MoveIntVector
		mov	bx,offset RomBootVect
		push	cs
		pop	es
		call	SetIntVector
RomBootInit_x:	popp	es,bx,ax
		ret
	endif
RomBootInit	endp

IntVectAddr	proc	near		; entry AL=vector
		xor	bx,bx		; exit: ES:BX=0:AL*4
		mov	es,bx		; other registers preserved
		mov	bl,al
		add	bx,bx
		add	bx,bx
		ret
IntVectAddr	endp

GetIntVector	proc	near		; Function 35h - get vector
		call	IntVectAddr	; entry AL=vector to get
GetIntVect_1:	les	bx,es:[bx]	; exit: ES:BX=vector value
		ret			; other registers preserved
GetIntVector	endp

SetIntVector	proc	near		; Set Interrupt Vector AL
		pushf			; entry AL=vector to set, ES:BX=value
		push	es		; exit: vector modified
		push	bx		; all registers preserved
		call	IntVectAddr
		cli
		pop	es:[bx]
		pop	es:[bx+2]
		popf
		jmp	GetIntVect_1
SetIntVector	endp

MoveIntVector	proc	near		; Move (copy) Interrupt Vector
		call	GetIntVector	; entry AL=vect to get, AH=vect to set
		xchg	al,ah		; exit: vector set, ES:BX=vector value
		call	SetIntVector	; other registers preserved
		xchg	al,ah
		ret
MoveIntVector	endp

TryDisk		proc	near		; returns ZF=1 on success
		mov	cx,4		; changes BX,CX; assumes ES=0, DX=disk
TryDiskLoop:	push	cx
		mov	ah,0
		int	int_biosdisk
		jc	TryDiskResFlt
		mov	cl,1
		mov	bx,BootLoadOffset
		mov	ax,biosdisk_read1
		int	int_biosdisk
		jc	TryDiskReadErr
		cmp	word ptr es:[bx+SectorSize-2],BootSign_1+256*BootSign_2
		pop	cx
		jmp	short TryDiskDone
TryDiskResFlt:	mov	ah,1
TryDiskReadErr:	pop	cx
		test	ah,disk_timeout
		loopz	TryDiskLoop
		test	bx,bx
TryDiskDone:	ret
TryDisk		endp

AskBootSelect	db	"Choose boot: digit=ROM",0
BootSelFloppy	db	", F=Floppy",0
BootSelHardDisk	db	", H=HardDisk",0

TwoChar		macro	c1,c2
		mov	al,c1
		call	BiosCharOut
		mov	al,c2
		call	BiosCharOut
		endm

RomBootVect	proc	far		; to be called by INT 19h
					; no return: boot or hang
	ifndef	TESTMODE
		mov	ax,(int_reboot shl 8)+int_bootsav
		call	MoveIntVector
		xor	bx,bx
		mov	es,bx
		call	SetIntVector
		inc	ax
		call	SetIntVector
	endif
		push	cs
		pop	ds
		mov	si,offset AskBootSelect
		call	ZStringOut
		xor	di,di		; flags - initially 0

		xor	dx,dx		; test if can boot from floppy A:
	ifdef	TESTMODE
		mov	bx,BootLoadSegment
		mov	es,bx
	endif
		call	TryDisk
		jnz	rbv_NoFloppy
		inc	di
		mov	si,offset BootSelFloppy
		call	ZStringOut
rbv_NoFloppy:	mov	dx,hard_disk	; test if can boot from hard disk
		mov	ah,diskf_query
		int	int_biosdisk
		jc	rbv_NoHardDisk
		test	dl,dl
		jz	rbv_NoHardDisk
		mov	dx,hard_disk
		call	TryDisk
		jnz	rbv_NoHardDisk
		inc	di
		inc	di
		mov	si,offset BootSelHardDisk
		call	ZStringOut
rbv_NoHardDisk:	mov	si,offset EndInitModule	; scan ROM images
		xor	cx,cx
rbv_CountRoms:	call	ShowNewLine
		lodsw
		inc	ax
		jz	rbv_RomsEnded
		dec	ax
		push	ax
		inc	cx
		mov	al,cl
		cmp	al,0ah
		sbb	al,69h
		das
		call	BiosCharOut
		TwoChar	'.',' '
		call	ZStringOut
		pop	si
		jmp	rbv_CountRoms
rbv_RomsEnded:	push	cx
		mov	ah,0
		int	int_biostime
		mov	bx,dx
rbv_WaitKey:	mov	ah,1
		int	int_bioskey
		jnz	rbv_GetKey
		mov	ah,0
		int	int_biostime
		sub	dx,bx
		mov	al,18
		mul	SelectTimeout
		dec	ax
		cmp	ax,dx
		jnb	rbv_WaitKey
		pop	cx
		test	di,1
		mov	al,'f'
		jnz	rbv_NoUCAlpha
		mov	al,'h'
		jcxz	rbv_NoUCAlpha
		mov	al,'1'
		jz	rbv_NoUCAlpha
rbv_GetKey:	pop	cx
		mov	ah,0
		int	int_bioskey
		cmp	al,'A'
		jb	rbv_NoUCAlpha
		cmp	al,'Z'
		ja	rbv_NoUCAlpha
		or	al,20h
rbv_NoUCAlpha:	xor	dx,dx
		cmp	al,'a'
		jb	rbv_A_NoAlpha
		cmp	al,'f'
		jnz	rbv_AnsNoFlop
		test	di,1
		jnz	rbv_AnsIsDisk
rbv_AnsNoFlop:	cmp	al,'h'
		jnz	rbv_AnsNoDisk
		test	di,2
		jz	rbv_AnsNoDisk
		mov	dl,hard_disk
rbv_AnsIsDisk:	call	TryDisk
		jnz	rbv_RomsEnded
		mov	cx,1
		push	es
		push	bx
		ret

rbv_AnsNoDisk:	sub	al,'a'-'9'-1
rbv_A_NoAlpha:	sub	al,'0'
		jbe	rbv_RomsEnded
		cmp	al,cl
		ja	rbv_RomsEnded
		mov	cl,al			; get selected ROM image
		call	ShowNewLine
		mov	ax,offset EndInitModule
rbv_get_ROM:	xchg	ax,si
		lodsw
		loop	rbv_get_ROM
		call	ZStringOut
		call	ShowNewLine
		xor	di,di
		mov	ax,RomLoadSegment
		mov	es,ax
rbv_get_ROM_1:	lodsw
		xchg	ax,cx
		jcxz	rbv_get_ROM_2
		rep	movsb
		lodsw
		xchg	ax,cx
		jcxz	rbv_get_ROM_2
		mov	al,es:[di-1]
		rep	stosb
		jmp	rbv_get_ROM_1
	ifdef	TESTMODE
rbv_get_ROM_2:	int	3
		jmp	$
	else
rbv_get_ROM_2:	push	cs
		call	rbv_call_ROM
rbv_get_ROM_e:	int	int_reboot
		sti
		jmp	rbv_get_ROM_e
rbv_call_ROM:	push	es
		mov	ax,3
		push	ax
		ret
	endif
RomBootVect	endp

ShowNewLine	proc	near
		mov	al,ch_cr
		call	BiosCharOut
		mov	al,ch_lf
		; fall int BiosCharOut
ShowNewLine	endp

BiosCharOut	proc	near		; outputs char from AL, changes AH
		mov	ah,biosvideo_wrtty
		push	bx
		mov	bx,biosvideo_yellow
		int	int_biosvideo
		pop	bx
		ret
BiosCharOut	endp

ZStringOut	proc	near		; Display 0-terminated string from SI
		push	ax		; destroys SI only
		cld
ZStringOut_l:	lodsb
		cmp	al,0
		jz	ZStringOut_x
		call	BiosCharOut
		jmp	ZStringOut_l
ZStringOut_x:	pop	ax
		ret
ZStringOut	endp

	ifdef	TESTMODE
		db	39 dup(-1)
	endif
EndInitModule	label	near
;RomPages	=	1+(offset EndInitModule-offset RomHeader-1)/PageSize
Text	ends
	end
