#
# Model Standardowy
#

class Particle:
	def __init__(self, full_name, name, mass=0, charge=0, color=False):
		self._full_name = full_name
		self._name = name 
		self._mass = mass 
		self._charge = charge 
		self._color = color
		self._baryon_number = 0
		self._lepton_number = 0
	def __str__(self):
		return self._full_name
	def __repr__(self):
		return f"{self.__class__.__name__}({self._full_name}, {self._name}, {self._mass}, {self._charge}, {self._color})"
	def get_baryon_number(self):
		return self._baryon_number
	def get_lepton_number(self):
		return self._lepton_number
	def get_antiparticle(self):
		name = self._name
		if '-' in name:
			name = name.replace('-','+')
		elif '+' in name:
			name = name.replace('+','-')
		else:
			name = name + '~'
		p = self.__class__(full_name='anti'+self._full_name, name=name, mass=self._mass, charge=-1*self._charge, color=self._color)
		p._baryon_number = p._baryon_number * -1
		p._lepton_number = p._lepton_number * -1
		return p

class Fermion(Particle):
	def __init__(self, **kwargs):
		super().__init__(**kwargs)
		self._spin = 0.5

class Scalar(Particle):
	def __init__(self, **kwargs):
		super().__init__(**kwargs)
		self._spin = 0.0

class Vector(Particle):
	def __init__(self, **kwargs):
		super().__init__(**kwargs)
		self._spin = 1.0

class Quark(Fermion):
	def __init__(self, **kwargs):
		super().__init__(**kwargs)
		self._color = True
		self._baryon_number = 1/3

class Lepton(Fermion):
	def __init__(self, **kwargs):
		super().__init__(**kwargs)
		self._lepton_number = 1


# BSM Physics
class Leptoquark(Quark, Lepton):
	def __init__(self, **kwargs):
		super().__init__(**kwargs)
	def __repr__(self):
		return f"Leptoquark({self._full_name}, {self._name}, {self._mass}, {self._charge})"

if __name__ == '__main__':
	MEV = 1e-3
	EV = 1e-9
	GEV = 1 
	# quark sector
	u = Quark(full_name='up', name='u', mass=2.2*MEV, charge=1/3)

	d = Quark(full_name='down', name='d', mass=4.7*MEV, charge=-2/3)
	c = Quark(full_name='charm', name='c', mass=1.28*GEV, charge=1/3)
	s = Quark(full_name='strange', name='s', mass=96*MEV, charge=-2/3)
	t = Quark(full_name='top', name='t', mass=173.1*GEV, charge=1/3)
	b = Quark(full_name='bottom', name='b', mass=4.18*GEV, charge=-2/3)
	# lepton sector
	e = Lepton(full_name='electron', name='e-', mass=0.511*MEV, charge=-1)
	ve = Lepton(full_name='neutrino (electron)', name='ve', mass=0.0, charge=0.0)
	mu = Lepton(full_name='muon', name='mu-', mass=105.66*MEV, charge=-1)
	vmu = Lepton(full_name='neutrino (muon)', name='vmu', mass=0.0, charge=0.0)
	tau = Lepton(full_name='tau', name='tau-', mass=1.7768*GEV, charge=-1)
	vtau = Lepton(full_name='neutrino (tau)', name='vtau', mass=0.0, charge=0.0)
	# gauge bosons
	g = Vector(full_name='gluon', name='g', mass=0.0, charge=0.0, color=True)
	a = Vector(full_name='photon', name='a', mass=0.0, charge=0.0, color=False)
	z = Vector(full_name='Z0', name='z', mass=91.19*GEV, charge=0.0, color=False)
	wp = Vector(full_name='W', name='w+', mass=80.39*GEV, charge=1.0, color=False)
	# Higgs sector
	h = Scalar(full_name='Higgs', name='h', mass=124.97*GEV, charge=0.0, color=False)

	SM = [u,d,c,s,t,b,e,ve,mu,vmu,tau,vtau,g,a,z,wp,h]
	antiSM = []
	for part in SM:
		antipart = part.get_antiparticle()
		antiSM.append(antipart)

	SM = SM+antiSM

	for part in SM:
		print('-'*40)
		print(part)
		print('lepton no: {}\t baryon no: {}'.format(part.get_lepton_number(), part.get_baryon_number()))
		print(part.__repr__())

	print(Leptoquark.__mro__)
	lq = Leptoquark(full_name='leptoquark', name='lq', mass=500*GEV, charge=1/3)
	print('#'*40)
	print(lq)
	print(lq.__repr__())
	print('baryon number')
	print(lq.get_baryon_number())
	print('lepton number')
	print(lq.get_lepton_number())
	print('lepton_color')
	print(lq._color)

	
