c     PROGRAMS FOR ONE-LOOP ON-SHELL CALCULATIONS IN THE MSSM
c     Authors: P.H.Chankowski, S.Pokorski, J.Rosiek
c     e-mail: rosiek@fuw.edu.pl
c             chank@padova.infn.it
 
c     FILENAME: ZZS_CRV.FOR
c     Released: 1: 7:1993(J.R.)
 
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     Evaluation of the cross section for the Bjorken process       c
c     e^-e^+ --> Z0* S_i --> ff S_i at the CMS energy = sqrt(s0)    c
c     (version for Z0 produced off-shell). Function zzs_cr_virt     c
c     outputs the total cross section in [picobarns] for the final  c
c     state S_i + fermion pair summed over fermion species          c
c     Array crf return cross section for each fermion type          c
c     in the final state produced via virtual Z* or soft photon     c
c     exchange (for the explanation see comments below)             c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 
      double precision function zzs_cr_virt(i,s0,crf,errin,errout,niter)
c     Cross sections for e+e- -> SZ* -> Sff production
c     Simple Romberg procedure integrates matrix elements over allowed
c     phase space
c     zzs_cr_virt: total cross section summed over fermion species
c     i:           number of scalar produced
c     s0:          Total CMS energy of electron pair
c     crf(if):     cross section ee -> Z*S -> ffS for given fermion type
c       1  < if < 3   neutrinos
c       4  < if < 6   leptons
c       7  < if < 9   u-quarks
c       10 < if < 12  d-squarks
c     errin:       assumed maximal error of integration
c     errout:      real (achieved) error of integration for each fermion type
c     niter:       maximal number of iterations. Time of calculations
c                  is proportional to 2^niter
c     This procedure has been designed basically to calculate e+e- -> Sff
c     cross section for LEP 100 energy s0 = MZ^2. It should work also for
c     other energies, but simplifications made can lead to some errors
c     for s0 far from resonance (of the order of alpha_EM/(4 pi) times
c     ratio of photon to Z propagator).
c     Another possible source of error is connected with the virtual
c     pseudoscalar/neutral Goldstone production in the intermediate step,
c     instead of Z*, or the e+e- -> ff -> Sff process (we neglect respective
c     terms in the expressions for cross sections). This error can be visible
c     only for e+e- -> Sbb (also Stau+tau- for large tan(beta)) cross section,
c     because of the weak coupling of Higgs bosons to the light fermions.

      implicit double precision (a-h,o-z)
      parameter (maxint=20,nf=12,res=7225)
      logical zp_stat
      dimension crf(nf),zero(nf),tmp(nf),errout(nf)
      common/hm_phys/frm(2),frm2(2)
      common/rombdat/q(maxint)
      common/vpar/st,ct,st2,ct2,sct,sct2,e,e2,alpha,wm,wm2,zm,zm2,pi,sq2
      common/zpeak/zm0,gam,zp_stat
      zzs_cr_virt = 0
      do 10 j=1,nf
10      zero(j) = 0
c     Check if real scalar production possible
      if (frm(i).ge.sqrt(s0)) return
c     Initialize common factors
      call crv_form_init(s0,i)
c     Number of iteration (maximum 10)
      if (niter.le.10) then
        nit = niter
      else
        nit = 10
      end if
c     Integration limits. If upper limit too close to Z mass, divide
c     integral into singular and non singular part (limit: b > 85^2)
      a = 0
      b = (sqrt(s0) - frm(i))**2
      zp_stat = .false.
      if (b.gt.res) then
c     Calculate cross sections in the common point
        call crv_form(res,s0,i)
        do 20 j=1,nf
20        tmp(j) = crv_ferm(j,res,s0,i)
c     Integral over non-singular range
        call romberg(i,s0,a,res,zero,tmp,crf,errin/2,errout,nit)
c     Integral over singular range (result and error stored temporary
c     in arrays zero and tmp)
        zp_stat = .true.
        gam = gam_z_0()
        zm0 = zm
        fmin = atan((res - zm2)/gam/zm)
        fmax = atan((b - zm2)/gam/zm)
        fact = (res - zm2)**2/zm/gam + zm*gam
        do 30 j=1,nf
30        tmp(j) = fact*tmp(j)
        call romberg(i,s0,fmin,fmax,tmp,zero,zero,errin/2,tmp,nit)
c     Add results and errors
        do 40 j=1,nf
          errout(j) = sqrt((errout(j)*crf(j))**2 + (zero(j)*tmp(j))**2)
          crf(j) = crf(j) + zero(j)
40        if (crf(j).ne.0) errout(j) = errout(j)/crf(j)
      else
c       Integral only over non-singular range
        call romberg(i,s0,a,b,zero,zero,crf,errin,errout,nit)
      end if
c     Total cross section (summed over fermions)
      do 50 j=1,nf
50      zzs_cr_virt = zzs_cr_virt + crf(j)
c     release interpolation arrays
      do j=1,6
        call mh_reset(j)
      end do
      return
      end

      subroutine crv_form_init(s0,i)
c     common invariants for e+e- -> SZ* -> Sff cross setion
c     independent on Z* energy
      implicit double precision (a-h,o-z)
      parameter(npit=6)
      logical mh_stat
      complex*16 zt_ren,fzt_ren,ft_ren
      complex*16 zt,ft,fzt
      complex*16 zzs(6)
      complex*16 a1,a2,b1,b2,c1,c2,dz,ze,zg
      complex*16 zms,zmp
      dimension sx(npit),fgr(npit),fgi(npit),fpr(npit),fpi(npit),
     1                   fqr(npit),fqi(npit)
      common/vpar/st,ct,st2,ct2,sct,sct2,e,e2,alpha,wm,wm2,zm,zm2,pi,sq2
      common/zext/zs(2),zms(2,2),zp(2),zmp(2,2),zz,zw,zeps
      common/crv_inv/a1,a2,b1,b2,c1,c2,ze,zg,dz,fact,alam
      common/hm_phys/frm(2),frm2(2)
c     check first if interpolation arrays are free
      do j=1,6
        if (mh_stat(j)) stop 'zzs_crv: mh_grid array not free'
      end do
c     variables not depending on s (s0 only)
      zt  = zt_ren(s0)
      ft  = ft_ren(s0)
      fzt = fzt_ren(s0)
      dz = s0 - zm2 - zt - fzt**2/(s0 - ft)
      ze = e2/2/sct2*(cr(i) + cr(3 - i)*zms(i,2))*zs(i)*zz
      zg = ze*fzt/(s0 - ft - fzt**2/(s0 - zm2 - zt))
c     Interpolate formfactor dependence on s to speed up numerical
c     integration
      delta = 9.d-4
      smax = (1 - delta)*(sqrt(s0) - frm(i))**2
      smin = delta*smax
      step = (smax - smin)/(npit - 1)
      do 10 j=1,npit
        sx(j) = smin + step*(j - 1)
        p = sx(j)
        q = frm2(i)
        pq = (s0 - p - q)/2
        call zzs_ren(p,q,pq,i,zzs)
        fgr(j) = dble(zzs(1))
        fgi(j) = dimag(zzs(1))
        fpr(j) = dble(zzs(3) - zzs(4))
        fpi(j) = dimag(zzs(3) - zzs(4))
        fqr(j) = dble(zzs(2) - zzs(5))
        fqi(j) = dimag(zzs(2) - zzs(5))
        call zzs_ren(p,q,pq,3 - i,zzs)
        fgr(j) = zz*zs(i)*(fgr(j) + dble(zms(i,2)*zzs(1)))
        fgi(j) = zz*zs(i)*(fgi(j) + dimag(zms(i,2)*zzs(1)))
        fpr(j) = zz*zs(i)*(fpr(j) + dble(zms(i,2)*(zzs(3) - zzs(4))))
        fpi(j) = zz*zs(i)*(fpi(j) + dimag(zms(i,2)*(zzs(3) - zzs(4))))
        fqr(j) = zz*zs(i)*(fqr(j) + dble(zms(i,2)*(zzs(2) - zzs(5))))
10      fqi(j) = zz*zs(i)*(fqi(j) + dimag(zms(i,2)*(zzs(2) - zzs(5))))
   
      call mh_grid(sx,fgr,npit,ierr,1)
      call mh_grid(sx,fgi,npit,ierr,2)
      call mh_grid(sx,fpr,npit,ierr,3)
      call mh_grid(sx,fpi,npit,ierr,4)
      call mh_grid(sx,fqr,npit,ierr,5)
      call mh_grid(sx,fqi,npit,ierr,6)
      return
      end

      subroutine crv_form(s,s0,i)
c     Common invariants for e+e- -> SZ* -> Sff cross section
c     dependent on Z* energy
      implicit double precision (a-h,o-z)
      parameter(ibr=10)
      complex*16 p_ren,zt_ren,fzt_ren,ft_ren,pz_ren
      complex*16 zt,ft,fzt
      complex*16 a1,a2,b1,b2,c1,c2,dz,ze,zg
      complex*16 dzv,ps(2),psmix
c     complex*16 dfv
      complex*16 fg,fp,fq
      complex*16 zms,zmp
      dimension vr(ibr),vi(ibr),nb(ibr)
      common/zext/zs(2),zms(2,2),zp(2),zmp(2,2),zz,zw,zeps
      common/vpar/st,ct,st2,ct2,sct,sct2,e,e2,alpha,wm,wm2,zm,zm2,pi,sq2
      common/hmass/cm(2),rm(2),pm(2),zr(2,2),zh(2,2)
      common/hm_phys/frm(2),frm2(2)
      common/crdat/pbarn,a,v
      common/crv_inv/a1,a2,b1,b2,c1,c2,ze,zg,dz,fact,alam

      alam = s0*s0 - 2*s0*(s + frm2(i)) + (s - frm2(i))**2
      if (alam/s0/s0.lt.1.d-6) then
        alam = 0
        return
      end if
      alam = sqrt(alam)

      zt  = zt_ren(s)
      ft  = ft_ren(s)
      fzt = fzt_ren(s)
c      dfv = s - ft - fzt**2/(s - zm2 - zt)
      dzv = s - zm2 - zt - fzt**2/(s - ft)

      call mh_interp(s,vr,nb,nbmax,1)
      call mh_interp(s,vi,nb,nbmax,2)
      fg = cmplx(vr(1),vi(1))
      call mh_interp(s,vr,nb,nbmax,3)
      call mh_interp(s,vi,nb,nbmax,4)
      fp = cmplx(vr(1),vi(1))
      call mh_interp(s,vr,nb,nbmax,5)
      call mh_interp(s,vi,nb,nbmax,6)
      fq = cmplx(vr(1),vi(1))

      c_add = 0
      ps(1) = p_ren(s,1,1)
      ps(2) = p_ren(s,2,2)
      psmix = p_ren(s,1,2)
      do 10 j=1,2
        k = 3 - j
10      c_add = c_add + (am(i,j) + zms(i,2)*am(3 - i,j))
     1        *pz_ren(s,j)/(s - pm(j)*pm(j) - ps(j)
     2        - psmix*psmix/(s - pm(k)*pm(k) - ps(k)))
      c_add = e*zz*zs(i)/sct*c_add

      a1 = v*(ze + fg) + zg
      b1 = v*fp
      c1 = v*(fq - c_add)
      a2 = a*(ze + fg)
      b2 = a*fp
      c3 = a*(fq - c_add)
      fact = abs(e2/dz/dzv)**2

      return
      end

      double precision function crv_ferm(if,s,s0,i)
c     (matrix element)x(phase space) for e+e- -> SZ* -> Sff cross setion
c     for fermion type defined in variable "if":
c     1  < if < 3   neutrinos
c     4  < if < 6   leptons
c     7  < if < 9   u-quarks
c     10 < if < 12  d-squarks
c     Soft photon contribution subtracted
      implicit double precision (a-h,o-z)
      complex*16 a1,a2,b1,b2,c1,c2,ze,zg,dz
      common/vpar/st,ct,st2,ct2,sct,sct2,e,e2,alpha,wm,wm2,zm,zm2,pi,sq2
      common/hm_phys/frm(2),frm2(2)
      common/crdat/pbarn,ae,ve
      common/fvert/q(4),v(4),a(4),nc(4)
      common/fmass/em(3),um(3),dm(3)
      common/crv_inv/a1,a2,b1,b2,c1,c2,ze,zg,dz,fact,alam

      crv_ferm = 0
      if ((if.lt.1).or.(if.gt.12).or.(alam.eq.0.d0)) return

c     fermion type (v,l,u,d) and generation (1,2,3)
      iftype = (if - 1)/3 + 1
      ifgen  = if - 3*(iftype - 1)
c     fermion mass
      if (iftype.eq.1) then
        fm = 0
      else if (iftype.eq.2) then
        fm = em(ifgen)
      else if (iftype.eq.3) then
        fm = um(ifgen)
      else
        fm = dm(ifgen)
      end if
c     s too small to produce ff pair
      if (s.le.4*fm*fm) return
c     fermion couplings
      fq = q(iftype)
      fv = v(iftype)
      fa = a(iftype)
      fnc = nc(iftype)

      en = s0 + s - frm2(i)

      t1 = 4*s0*(abs(a1)**2 + abs(a2)**2)
      t2 = (abs(2*a1 - en*b1)**2 + abs(2*a2 - en*b2)**2
     1   - 4*s*s0*(abs(b1)**2 + abs(b2)**2))*alam*alam/12/s
      t3 = dble((2*a1 + s*b1 + frm2(i)*c1)*conjg(b1 + c1)
     1        + (2*a2 + s*b2 + frm2(i)*c2)*conjg(b2 + c2)
     2        - s0*(b1*conjg(c1) + b2*conjg(c2)))*alam*alam/2

      am = fact*alam*sqrt(1 - 4*fm*fm/s)*(fv*fv*(s + 2*fm*fm)*(t1 + t2)
     1   + fa*fa*((s - 4*fm*fm)*t1 + (s + 2*fm*fm)*t2 + 4*fm*fm*t3))
      crv_ferm = pbarn/768/pi/pi/pi/s0/s0*fnc*am
      return
      end

      double precision function gam_z_0()
c     Z0 decay width (tree level EW + 3rd order QCD corrections included)
      implicit double precision (a-h,o-z)
      logical zstat
      complex*16 zt_ren,ft_ren,fzt_ren
      common/zwidth/z_gam,zfact,zstat
      common/vpar/st,ct,st2,ct2,sct,sct2,e,e2,alpha,wm,wm2,zm,zm2,pi,sq2
      if (.not.zstat) then
        al = alfas(zm)/pi
        zfact = 1 + al*(1 + al*(1.40923d0 - al*12.76706d0))
        z_gam = - dimag(zt_ren(zm2)
     1               + fzt_ren(zm2)**2/(zm2 - ft_ren(zm2)))/zm
        zfact = 1
        zstat = .true.
      end if
      gam_z_0 = z_gam
      return
      end

      subroutine romberg(i,s0,a,b,fa,fb,aint,errin,errout,niter)
c     Numerical integration (Romberg procedure)
      implicit double precision (a-h,o-z)
      parameter (maxint=20,nf=12)
      logical zp_stat
      dimension aint(nf),errout(nf),fa(nf),fb(nf)
      dimension w(nf,maxint),oldint(nf),pint(nf)
      common/rombdat/q(maxint)
      common/zpeak/zm,gam,zp_stat
      external init_romberg
c     First approximation of integral; initialize w, aint and oldint arrays
      x = (a + b)/2
      if (zp_stat) then
        x = zm*(gam*tan((a + b)/2) + zm)
        fac = (x - zm*zm)**2/zm/gam + zm*gam
      else
        fac = 1
      end if
      call crv_form(x,s0,i)
      do 10 j=1,nf
        aint(j) = 0
        oldint(j) = (b - a)/4*(fa(j) + 2*fac*crv_ferm(j,x,s0,i) + fb(j))
10      w(j,1) = oldint(j)
c     Next iterations; quit if error lower then assumed or maximal
c     number of iteration achieved
      do 20 j=2,niter
        call trapez(i,s0,a,b,j,pint,oldint)
        do 30 k=1,nf
          oldint(k) = pint(k)
30        call richards(w,pint,j,errout,aint,q,k)
c     Quit if error small enough
        error = errout(1)
        do 40 k=2,nf
40        error = max(errout(k),error)
20      if (error.le.errin) return
      return
      end

      subroutine trapez(i,s0,a,b,j,pint,oldint)
c     Numerical integration (trapez method)
      implicit double precision (a-h,o-z)
      parameter (maxint=20,nf=12)
      logical zp_stat
      dimension oldint(nf),pint(nf)
      common/zpeak/zm,gam,zp_stat
      n = 2**j
      h = (b - a)/n
      do 10 k=1,nf
10      pint(k) = 0.d0
      do 20 l=1,n-1,2
        x = a + h*l
        if (zp_stat) then
          x = zm*(gam*tan(x) + zm)
          fact = (x - zm*zm)**2/zm/gam + zm*gam
        else
          fact = 1
        end if
        call crv_form(x,s0,i)
        do 20 k=1,nf
20        pint(k) = pint(k) + fact*crv_ferm(k,x,s0,i)
      do 30 k=1,nf
30      pint(k) = oldint(k)/2 + h*pint(k)
      return
      end

      subroutine richards(w,pint,i,err,aint,q,j)
c     Richardson extrapolation
      implicit double precision (a-h,o-z)
      parameter (maxint=20,nf=12)
      dimension w(nf,maxint),q(maxint)
      dimension pint(nf),aint(nf),err(nf)
      a = w(j,1)
      b = a
      w(j,1) = pint(j)
      do 10 n=1,i-1
        if (n.lt.i-1) b = w(j,n + 1)
        w(j,n + 1) = w(j,n) + q(n)*(w(j,n) - a)
10      a = b
      aint(j) = w(j,i)
      if (aint(j).ne.0.d0) then
        err(j) = abs((w(j,i - 1) - a)/aint(j))
      else
        err(j) = abs(w(j,i - 1) - a)
      end if
      return
      end

      block data init_romberg
      implicit double precision (a-h,o-z)
      parameter (maxint=20)
      common/rombdat/q(maxint)
      data q/3.333333333333333d-001,6.666666666666667d-002,
     1       1.587301587301587d-002,3.921568627450980d-003,
     2       9.775171065493646d-004,2.442002442002442d-004,
     3       6.103888176768602d-005,1.525902189669642d-005,
     4       3.814711817595740d-006,9.536752259018191d-007,
     5       2.384186359449949d-007,5.960464832810452d-008,
     6       1.490116141589226d-008,3.725290312339702d-009,
     7       9.313225754828403d-010,2.328306437080797d-010,
     8       5.820766091685554d-011,1.455191522857861d-011,
     9       3.637978807104948d-012,9.094947017737554E-013/
      end



